using System;
using System.Collections;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Windows.Forms;

using System.Management;

namespace JacekMatulewski.Util
{
	/// <summary>
	/// Komponent prezentujcy ilo wolnego miejsca na wybranym dysku.
	/// </summary>
	/// <remarks>
	/// Klasa moe by uywana zarwno jako komponent wizualny, prezentuje wwczas ilo wolnego miejsca na wskazanym dysku w postaci paska postpu, lub jako komponent dostarczajcy szczegowych informacji o wybranym dysku. Su do tego wasnoci <b>DiskFreeSpace</b>, <b>DiskOccupiedSpace</b>, <b>DiskSize</b>, <b>DriveDescription</b>, <b>DriveFileSystem</b>, <b>DriveVolumeName</b> i <b>DriveVolumeSerialNumber</b>.
	/// Bibliotek <see cref="JacekMatulewski.Util">JacekMatulewski.Util</see> w skad ktrej wchodzi komponent <b>DiskFreeBar</b> mona cign ze strony <see href="http://codecentral.borland.com">Borland Codecental</see>, ID: 21816.
	/// </remarks>
	/// <example>
	/// Oto przykad dynamicznego stworzenia obiektu klasy <b>DiskFreeBar</b> pokazujcy ilo wolnego miejsca na dysku C:. Rozmiar komponentu jest dostosowany do szerokoci komponentu - rodzica (<b>this</b>).
	/// <code>
	/// DiskFreeBar dfb=new DiskFreeBar();
	/// dfb.Location=new Point(10,10);
	/// dfb.Size=new Size(this.ClientRectangle.Width-2*dfb.Location.X,30);
	/// dfb.DriveLetter='C';
	/// dfb.Parent=this;</code>
	/// </example>
	public class DiskFreeBar : System.Windows.Forms.UserControl
	{
		/// <summary>
		/// Required designer variable.
		/// </summary>
		private System.ComponentModel.Container components = null;
		private System.Windows.Forms.ProgressBar progressBar1;

		/// <summary>
		/// Tworzy now instancj klasy <b>DiskFreeBar</b>.
		/// </summary>
		public DiskFreeBar()
		{
			// This call is required by the Windows.Forms Designer.
			InitializeComponent();

			UstawPozycjePaska();
		}

		/// <summary>
		/// Czyci wszystkie zasoby uywane przez obiekt.
		/// </summary>
		protected override void Dispose(bool disposing)
		{
			if (disposing)
			{
				if (components != null)
					components.Dispose();
			}
			base.Dispose(disposing);
		}

		#region Component Designer generated code
		/// <summary>
		/// Required method for Designer support - do not modify 
		/// the contents of this method with the code editor.
		/// </summary>
		private void InitializeComponent()
		{
			this.progressBar1 = new System.Windows.Forms.ProgressBar();
			this.SuspendLayout();
			// 
			// progressBar1
			// 
			this.progressBar1.Dock = System.Windows.Forms.DockStyle.Fill;
			this.progressBar1.Location = new System.Drawing.Point(0, 0);
			this.progressBar1.Name = "progressBar1";
			this.progressBar1.Size = new System.Drawing.Size(304, 72);
			this.progressBar1.TabIndex = 0;
			// 
			// DiskFreeBar
			// 
			this.Controls.Add(this.progressBar1);
			this.Name = "DiskFreeBar";
			this.Size = new System.Drawing.Size(304, 72);
			this.ResumeLayout(false);
		}
		#endregion

		/// <summary>
		/// Pobiera wielko wolnego, cakowitego miejsca na wskazanym dysku.
		/// </summary>
		/// <returns>
		/// Liczb bajtw wolnego lub cakowitego miejsca na wskazanym dysku.
		/// </returns>
		/// <param name="driveLetter">
		/// Sama litera dysku (bez znakw ":\")
		/// </param>
		/// <param name="desiredInformation">
		/// W zalenoci od wartoci argumentu <paramref name="desiredInformation"></paramref> ("Size" lub "FreeSpace") zwracany jest cakowity rozmiar na dysku lub ilo wolnego miejsca. Moliwe wartoci acucha odpowiadaj indeksatorom klasy <see cref="System.Management.ManagementObject">System.Management.ManagementObject</see>.
		/// </param>
		/// <remarks>
		/// Wartoci zwracane przez t metod mog by atwiej odczytane z wasnoci <b>DiskSize</b>, <b>DiskFreeSpace</b>, <b>DiskOccupiedSpace</b>.
		/// </remarks>
		public static long GetDiskInfo(char driveLetter,string desiredInformation)
		{
			ManagementObject mo = new ManagementObject("Win32_LogicalDisk='"+driveLetter+":'");
			return long.Parse(mo[desiredInformation].ToString());
		}

		private char literaDysku='C';
		private long rozmiar=0;
		private long wolne=0;
		private long zajete=0;

		private void UstawPozycjePaska()
		{
			rozmiar=GetDiskInfo(literaDysku,"Size");
			wolne=GetDiskInfo(literaDysku,"FreeSpace");
			zajete=rozmiar-wolne;
			progressBar1.Value=(int)(100*zajete/rozmiar);
		}

		/// <summary>
		/// Wasno okrelajca dysk, ktrego ilo wolnego miejsca wywietlana jest na komponencie
		/// </summary>
		/// <value>
		/// Litera dysku (pojedyczy znak typu <b>char</b>)
		/// </value>
		[
			Category("Drive Information"),
			Description("Determines letter of drive which parameters are displayed.")
		]
		public char DriveLetter
		{
			set
			{
				literaDysku=char.ToUpper(value);
				UstawPozycjePaska();
			}
			get
			{
				return char.ToUpper(literaDysku);
			}
		}

		/// <summary>
		/// Odwiea komponent pobierajc aktualn ilo wolnego miejsca na dysku.
		/// </summary>
		public override void Refresh()
		{
		base.Refresh();
		UstawPozycjePaska();
		}

		/// <summary>
		/// Udostpnia cakowity rozmiar dysku wskazanego przez wasno <b>DriveLetter</b>.
		/// </summary>
		/// <value>
		/// Cakowity rozmiar dysku w bajtach
		/// </value>
		[
			Category("Drive Information"),
			Description("The DiskSize exposes disk size of drive determined by DriveLetter.")
		]
		public long DiskSize
		{
			get
			{
				return rozmiar;
			}
		}

		/// <summary>
		/// Udostpnia ilo wolnego miejsca na dysku wskazanym przez wasno <b>DriveLetter</b>.
		/// </summary>
		/// <value>
		/// Ilo wolnego miejsca na dysku w bajtach
		/// </value>
		[
			Category("Drive Information"),
			Description("The DiskFreeSpace exposes disk free space of drive determined by DriveLetter.")
		]
		public long DiskFreeSpace
		{
			get
			{
				return wolne;
			}
		}

		/// <summary>
		/// Udostpnia ilo zajtego miejsca na dysku wskazanym przez wasno <b>DriveLetter</b>.
		/// </summary>
		/// <value>
		/// Ilo zajtego miejsca na dysku w bajtach
		/// </value>
		[
			Category("Drive Information"),
			Description("The DiskOccupiedSpace exposes disk occupied space of drive determined by DriveLetter.")
		]
		public long DiskOccupiedSpace
		{
			get
			{
				return zajete;
			}
		}

		/// <summary>
		/// Udostpnia nazw woluminu dysku wskazanego przez wasno <b>DriveLetter</b>.
		/// </summary>
		/// <value>
		/// acuch zawierajcy nazw woluminu
		/// </value>
		[
			Category("Drive Information"),
			Description("The DriveVolumeName exposes drive volume name.")
		]
		public string DriveVolumeName
		{
			get
			{
				ManagementObject mo = new ManagementObject("Win32_LogicalDisk='"+literaDysku+":'");
				return mo["VolumeName"].ToString();
			}
		}

		/// <summary>
		/// Udostpnia opis typu dysku (lokalny, sieciowy CD itp.) wskazanego przez wasno <b>DriveLetter</b>.
		/// </summary>
		/// <value>
		/// acuch zawierajcy opis typu dysku
		/// </value>
		[
			Category("Drive Information"),
			Description("The DriveDescription exposes the drive type description.")
		]
		public string DriveDescription
		{
			get
			{
				ManagementObject mo = new ManagementObject("Win32_LogicalDisk='"+literaDysku+":'");
				return mo["Description"].ToString();
			}
		}

		/// <summary>
		/// Udostpnia informacje o systemie plikw (FAT) na dysku wskazanym przez wasno <b>DriveLetter</b>.
		/// </summary>
		/// <value>
		/// acuch identyfikujcy system plikw
		/// </value>
		[
			Category("Drive Information"),
			Description("The DriveFileSystem exposes the drive file system name.")
		]
		public string DriveFileSystem
		{
			get
			{
				ManagementObject mo = new ManagementObject("Win32_LogicalDisk='"+literaDysku+":'");
				return mo["FileSystem"].ToString();
			}
		}

		/// <summary>
		/// Udostpnia numer seryjny dysku wskazanego przez wasno <b>DriveLetter</b>.
		/// </summary>
		/// <value>
		/// acuch zawierajcy numer seryjny dysku
		/// </value>
		[
			Category("Drive Information"),
			Description("The DriveVolumeSerialNumber exposes the drive volume serial number.")
		]
		public string DriveVolumeSerialNumber
		{
			get
			{
				ManagementObject mo = new ManagementObject("Win32_LogicalDisk='"+literaDysku+":'");
				return mo["VolumeSerialNumber"].ToString();
			}
		}
	}
}
