﻿using System;
using System.Net;
using System.Net.NetworkInformation;
using System.Net.Sockets;
using System.Text;
using System.Threading;

namespace Umk.Broadcast
{
    public class BroadcastHelper
    {
        private readonly int _port;
        private readonly string _currentDeviceName;
        private readonly Action<IPAddress, string> _deviceFoundHandler;

        private readonly int timeout = 1000;
        private Thread _senderThread, _receiverThread;
        private bool _running;
        private UdpClient _udpClient;

        public BroadcastHelper(int port, string currentDeviceName, Action<IPAddress, string> deviceFoundHandler)
        {
            _port = port;
            _currentDeviceName = currentDeviceName;
            _deviceFoundHandler = deviceFoundHandler;
        }

        public void Start()
        {
            _udpClient = new UdpClient(_port);
            _udpClient.Client.ReceiveTimeout = timeout;

            _running = true;

            _senderThread = new Thread(StartSending);
            _senderThread.Start();

            _receiverThread = new Thread(StartReceiving);
            _receiverThread.Start();
        }

        private void StartSending()
        {
            //UdpClient senderUdp = new UdpClient(port);

            string message = "ntpp:discovery";
            var messageBytes = Encoding.UTF8.GetBytes(message);

            while (_running)
            {
                //udpClient.Send(messageBytes, messageBytes.Length, new IPEndPoint(IPAddress.Broadcast, port));
                SendToEveryInterface(messageBytes);
                Thread.Sleep(timeout);
            }

            if (_udpClient != null)
            {
                _udpClient.Close();
                _udpClient = null;
            }
        }

        private void SendToEveryInterface(byte[] message)
        {
            IPEndPoint ip = new IPEndPoint(IPAddress.Broadcast, _port);
            NetworkInterface[] interfaces = NetworkInterface.GetAllNetworkInterfaces();
            foreach (NetworkInterface adapter in interfaces)
            {
                if (!adapter.Supports(NetworkInterfaceComponent.IPv4)) continue;
                try
                {
                    IPInterfaceProperties adapterProperties = adapter.GetIPProperties();
                    foreach (var ua in adapterProperties.UnicastAddresses)
                    {
                        if (ua.Address.AddressFamily != AddressFamily.InterNetwork) continue;
                        Socket broadcastSocket = new Socket(AddressFamily.InterNetwork, SocketType.Dgram, ProtocolType.Udp);
                        broadcastSocket.SetSocketOption(SocketOptionLevel.Socket, SocketOptionName.Broadcast, 1);
                        IPEndPoint localEndPoint = new IPEndPoint(ua.Address, _port);
                        broadcastSocket.Bind(localEndPoint);
                        broadcastSocket.SendTo(message, ip);
                        broadcastSocket.Close();
                    }
                }
                catch { }
            }
        }

        private void StartReceiving()
        {
            //UdpClient receiverUdp = new UdpClient(port);
            //receiverUdp.Client.ReceiveTimeout = 1000;

            IPEndPoint remoteEndPoint = new IPEndPoint(IPAddress.Any, _port);

            while (_running)
            {
                try
                {
                    byte[] messageBytes = _udpClient.Receive(ref remoteEndPoint);
                    string messageString = Encoding.UTF8.GetString(messageBytes);
                    var messageSplit = messageString.Split(new[] { ':' }, StringSplitOptions.RemoveEmptyEntries);

                    if (messageSplit[0].Equals("ntpp"))
                    {
                        if (messageSplit[1].Equals("discovery"))
                        {
                            string response = "ntpp:response:" + _currentDeviceName;
                            byte[] responseBytes = Encoding.UTF8.GetBytes(response);
                            _udpClient.Send(responseBytes, responseBytes.Length, remoteEndPoint);
                        }
                        else if (messageSplit[1].Equals("response"))
                        {
                            string remoteDeviceName = messageSplit[2];
                            if (remoteDeviceName != _currentDeviceName)
                                _deviceFoundHandler(remoteEndPoint.Address, remoteDeviceName);
                        }
                    }
                }
                catch { }
            }

            if (_udpClient != null)
            {
                _udpClient.Close();
                _udpClient = null;
            }
        }

        public void Stop()
        {
            _running = false;

            _senderThread.Join();
            _receiverThread.Join();
        }
    }
}
