﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

using Android.App;
using Android.Content;
using Android.Database;
using Android.Database.Sqlite;
using Android.OS;
using Android.Runtime;
using Android.Views;
using Android.Widget;
using SQLiteDatabase.DBModel;

namespace SQLiteDatabase
{
    public class GenericDatabase : SQLiteOpenHelper
    {
        private Dictionary<Type, ITable> _tables = new Dictionary<Type, ITable>();
        public string DBName { get; }

        public GenericDatabase(Context context, string dbName) : base(context, dbName, null, 1)
        {
            DBName = dbName;            
        }

        public void RegisterType<TClass>()
            where TClass : IClassTable, new()
        {
            var type = typeof(TClass);
            var table = new Table<TClass>();
            _tables.Add(type, table);
        }

        public override void OnCreate(Android.Database.Sqlite.SQLiteDatabase db)
        {
            foreach (var tablePair in _tables)
            {
                var table = tablePair.Value;
                string createCommand = $"create table {table.TableName} (";
                for (int columnNumber = 0; columnNumber < table.TableColumnNames.Length; columnNumber++)
                {
                    string columnName = table.TableColumnNames[columnNumber];
                    string columnProperty = table.TableColumnProperties[columnNumber];
                    createCommand += columnName + " " + columnProperty + ",";
                }
                createCommand = createCommand.TrimEnd(',');
                createCommand += ")";

                db.ExecSQL(createCommand);
            }
        }

        public override void OnUpgrade(Android.Database.Sqlite.SQLiteDatabase db, int oldVersion, int newVersion)
        {

        }

        private ContentValues ToContentValues(IClassTable tableClass)
        {
            ContentValues values = new ContentValues();
            var type = tableClass.GetType();
            var table = _tables[type];
            var properties = type.GetProperties();

            for (int i = 0; i < properties.Length; i++)
            {
                var propertyType = properties[i].PropertyType;
                var value = properties[i].GetValue(tableClass);
                if (propertyType == typeof(string))
                    values.Put(table.TableColumnNames[i], (string)value);
                else if (propertyType == typeof(int))
                    values.Put(table.TableColumnNames[i], (int)value);
            }

            return values;
        }

        public void AddRecord(IClassTable tableClass)
        {
            ContentValues values = ToContentValues(tableClass);
            var type = tableClass.GetType();
            var table = _tables[type];
            var db = WritableDatabase;
            db.InsertOrThrow(table.TableName, null, values);
        }

        private T EntityFromCarriage<T>(ICursor carriage)
            where T : IClassTable
        {
            var type = typeof(T);
            var properties = type.GetProperties();

            T entity = Activator.CreateInstance<T>();

            for (int i = 0; i < properties.Length; i++)
            {
                var property = properties[i];
                if (property.PropertyType == typeof(int))
                    property.SetValue(entity, carriage.GetInt(i));
                else if (property.PropertyType == typeof(string))
                    property.SetValue(entity, carriage.GetString(i));
            }

            return entity;
        }

        public T[] GetAllRecords<T>(string orderBy)
            where T : IClassTable
        {
            var db = ReadableDatabase;
            var type = typeof(T);
            var table = _tables[type];
            ICursor carriage = db.Query(table.TableName, table.TableColumnNames, null, null, null, null, orderBy);
            List<T> list = new List<T>();
            while (carriage.MoveToNext())
            {
                list.Add(EntityFromCarriage<T>(carriage));
            }

            carriage.Close();

            return list.ToArray();
        }

        public T[] GetAllRecords<T>()
            where T : IClassTable
        {
            return GetAllRecords<T>(null);
        }

        public T GetById<T>(int id)
            where T : IClassTable
        {
            var db = this.ReadableDatabase;
            T entity = GetAllRecords<T>().SingleOrDefault(x => x.Id == id);
            return entity;
        }

        public bool DeleteRecord<T>(int id)
            where T : IClassTable
        {
            Android.Database.Sqlite.SQLiteDatabase db = this.WritableDatabase;
            string[] whereArgs = { id.ToString() };
            var type = typeof(T);
            var table = _tables[type];
            return db.Delete(table.TableName, "Id=?", whereArgs) > 0;
        }

        public bool UpdateRecord<T>(int id, T entity)
            where T : IClassTable
        {
            Android.Database.Sqlite.SQLiteDatabase db = this.WritableDatabase;
            ContentValues values = ToContentValues(entity);
            string[] whereArgs = { id.ToString() };
            var type = typeof(T);
            var table = _tables[type];
            return db.Update(table.TableName, values, "Id=?", whereArgs) > 0;
        }
    }


}