unit Cellular_automation;

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, StdCtrls, Menus, ExtCtrls, ComCtrls, Math, ExtDlgs;

const cells_x = 60-1;
const cells_y = 60-1;

const board_width = 601;
const board_height = 601;
const board_pixelformat = pf24bit;

type Toptions = RECORD
  neighbourhood: (NEUMANN, MOORE);
  game_in_progress: boolean;
  step_mode: boolean;
  b: array[0..8] of byte; //birth condition
  d: array[0..8] of byte; //survive condition
end;

type Tinfo = RECORD
  cells:word;
  alive:word;
  dead:word;
  births:longint;
  deaths:longint;
  factor:real;
  steps:longint;
  changes:word;
end;

type table = array[0..cells_x]of array[0..cells_y] of byte;

//------------------------------------------------------------------------------

//Initialization of games board canvas
procedure init_board();

//Draws background rectangle on board
procedure clear_board();

//Draws cells on board respectively to cell state
procedure draw_on_board();

//Resets all cells values to 0 (dead)
procedure cells_clear();

//Sets/Unsets choosen (by user, via mouse) cell
procedure change_cell_state(x:integer; y:integer);

//Safe function to get cell value
function cell_val(x:integer; y:integer):byte;

//Returns number of cells neighbours (acc. to Moore)
function moore_neighbour(x:integer; y:integer):shortint;

//Returns number of cells neighbours (acc. to von Neumann)
function von_neumann_neighbour(x:integer; y:integer):shortint;

//Displays statistic info on additional form, etc.
procedure stat_info();

//Game engene - changes cell states
procedure game_of_life(); //<======

//Resets game state (counters, board, etc.)
procedure reset_game();

//Stops games timer
procedure pause_game();

//Resumes games timer
procedure resume_game();

//Copy cells from game board to temp board
procedure copy_board();

//Restores cells from temp board to game board
procedure restore_board();

//------------------------------------------------------------------------------

implementation

uses Unit1, Unit2;

procedure init_board();
begin
with Form1.Board.Picture.Bitmap do
begin
  Width := board_width;
  Height := board_height;
  PixelFormat := board_pixelformat;
end;
end;

procedure clear_board();
var color:Tcolor;
begin
with Form1.Board.Picture.Bitmap.Canvas do
  begin
  color := RGB(230,230,230);
  Pen.Color := color;
  Brush.Color := color;
  Rectangle(0,0,board_width,board_height);
end;
end;

procedure draw_on_board();
var i,j:integer;
begin
with Form1.Board.Picture.Bitmap.Canvas do
  begin
    for j := 0 to cells_y do
    for i := 0 to cells_x do
    begin
      if (cell[i][j] = 0) then
      begin
        Pen.Color := clWhite;
        Brush.Color := clWhite;
      end else
      begin
        Pen.Color := clBlack;
        Brush.Color := clBlack;
      end;
      Rectangle(10*i+1,10*j+1,10*i+10,10*j+10);
    end;
end;
end;

procedure cells_clear();
var i,j:byte;
begin
  for j := 0 to cells_y do
  for i := 0 to cells_x do cell[i][j] := 0;
end;

procedure change_cell_state(x:integer; y:integer);
begin
  if (x >= 0) AND (y >= 0) AND (x <= cells_x) AND (y <= cells_y) then
  begin
    if (cell[x][y] = 0) then cell[x][y] := 1 else cell[x][y] := 0;
  end;
end;

function cell_val(x:integer; y:integer):byte;
begin
  if (x < 0) OR (y < 0) OR (x > cells_x) OR (y > cells_y) then
  cell_val := 0 else cell_val := cell[x][y];
end;

function moore_neighbour(x:integer; y:integer):shortint;
var n:byte; //Neighbours counter
begin
    n := 0;
    n := n + cell_val(x+0,y-1);
    n := n + cell_val(x+1,y+0);
    n := n + cell_val(x+0,y+1);
    n := n + cell_val(x-1,y+0);
    moore_neighbour := n;
end;

function von_neumann_neighbour(x:integer; y:integer):shortint;
var n:byte; //Neighbours counter
begin
    n := 0;
    n := n + cell_val(x+0,y-1);
    n := n + cell_val(x+1,y-1);
    n := n + cell_val(x+1,y+0);
    n := n + cell_val(x+1,y+1);
    n := n + cell_val(x+0,y+1);
    n := n + cell_val(x-1,y+1);
    n := n + cell_val(x-1,y+0);
    n := n + cell_val(x-1,y-1);
    von_neumann_neighbour := n;
end;

procedure stat_info();
begin
  with Info_Form do
  begin
    Label9.Caption  := IntToStr(info.cells);
    Label10.Caption := IntToStr(info.alive);
    Label11.Caption := IntToStr(info.dead);
    Label12.Caption := IntToStr(info.births);
    Label13.Caption := IntToStr(info.deaths);
    Label14.Caption := FloatToStrF(info.factor,ffFixed,5,5);
    Label15.Caption := IntToStr(info.steps);
    Label16.Caption := IntToStr(info.changes);
  end;

  Form1.StatusBar.Panels[2].Text := 'Step: '+IntToStr(info.steps);
end;

procedure game_of_life();
var i,j:byte;
var neighbours:shortint;
var temp:table;
begin
  info.changes := 0;

  for j:=0 to cells_y do
  for i:=0 to cells_x do
  begin
    if (options.neighbourhood = NEUMANN) then
      neighbours := von_neumann_neighbour(i,j)
    else
      neighbours := moore_neighbour(i,j);

    if (cell[i][j] = 0) AND (options.b[neighbours] = 1) then
    begin
      //Birth of cell
      temp[i][j] := 1;
      info.births := info.births + 1;
      info.changes := info.changes + 1;
    end
    else if (cell[i][j] = 1) AND (options.d[neighbours] = 0) then
    begin
      //Death of cell
      temp[i][j] := 0;
      info.deaths := info.deaths + 1;
      info.changes := info.changes + 1;
    end else
    temp[i][j] := cell[i][j];
  end;

  info.dead  := 0;
  info.alive := 0;
  for j:=0 to cells_y do
  for i:=0 to cells_x do
  begin
    cell[i][j] := temp [i][j];
    if (cell[i][j] = 1) then info.alive := info.alive + 1
    else info.dead := info.dead + 1;
  end;
  info.cells := info.alive + info.dead;
  if (info.deaths > 0) then info.factor := info.births/info.deaths else info.factor := 0.0;
  info.steps := info.steps + 1;
end;

procedure reset_game();
begin
  pause_game();

  cells_clear();
  clear_board();
  draw_on_board();

  with info do
  begin
    alive  := 0;
    dead   := 3600;
    cells  := dead - alive;
    births := 0;
    deaths := 0;
    factor := 0.0;
    steps  := 0;
    changes := 0;
  end;
  stat_info();

  options.game_in_progress := False;
end;

procedure pause_game();
begin
  Form1.Timer.Enabled := False;
  Form1.Start_Button.Caption := 'Start';
end;

procedure resume_game();
begin
  Form1.Timer.Enabled := True;
  Form1.Start_Button.Caption := 'Pause';
end;

procedure copy_board();
var i,j:byte;
begin
  for j:=0 to cells_y do
  for i:=0 to cells_x do copy[i][j] := cell[i][j];
end;

procedure restore_board();
var i,j:byte;
begin
  for j:=0 to cells_y do
  for i:=0 to cells_x do cell[i][j] := copy[i][j];
end;

end.
