﻿using System;
using System.Collections.Generic;
using System.Drawing;
using System.Windows.Forms;

namespace LogisticGrowthEquation
{
    public struct Point2d
    {
        public double x, y;
    }

    public partial class DrawGraphForm : Form
    {
        private double xMin;
        private double xMax;
        private double yMin;
        private double yMax;
        private int dot, n;
        private string title;
        private Brush brush;
        private Brush potBrush;
        private Font font;
        private Pen pen1, pen2;
        private List<Point2d> pts;

        private void FindMinMax()
        {
            xMin = yMin = double.MaxValue;
            xMax = yMax = double.MinValue;

            for (int i = 0; i < n; i++)
            {
                Point2d pt = pts[i];
                double x = pt.x;
                double y = pt.y;

                if (x < xMin)
                    xMin = x;
                if (x > xMax)
                    xMax = x;
                if (y < yMin)
                    yMin = y;
                if (y > yMax)
                    yMax = y;
            }
        }

        public DrawGraphForm(
            int dot,
            string title,
            List<Point2d> pts)
        {
            InitializeComponent();
            this.dot = dot;
            this.title = title;
            this.pts = pts;
            n = pts.Count;
            FindMinMax();
            brush = new SolidBrush(Color.Black);
            potBrush = new SolidBrush(Color.Red);
            pen1 = new Pen(Color.Black);
            pen2 = new Pen(Color.Blue);
            font = new Font("Verdana", 8f, FontStyle.Regular);
            panel1.Paint += new PaintEventHandler(PanelPaintHandler);
        }

        private void DrawGraph(float u0, float v0,
            float u1, float v1, Graphics g)
        {
            try
            {
                float xMin = u0;
                float yMin = v0;
                float xMax = u1;
                float yMax = v1;

                float xSpan = xMax - xMin;
                float ySpan = yMax - yMin;

                float deltaX = xSpan / 8.0f;
                float deltaY = ySpan / 8.0f;
                float height = panel1.Height;
                float width = panel1.Width;

                float sx0 = 2f * width / 16f;
                float sx1 = 14f * width / 16f;
                float sy0 = 2f * height / 16f;
                float sy1 = 14f * height / 16f;

                float xSlope = (sx1 - sx0) / xSpan;
                float xInter = sx0 - xSlope * xMin;
                float ySlope = (sy0 - sy1) / ySpan;
                float yInter = sy0 - ySlope * yMax;

                float x = xMin;
                float y = yMin;

                string fTitle = "Wykres wzrostu populacji";

                float w = g.MeasureString(fTitle, font).Width;
                float h = g.MeasureString(fTitle, font).Height;

                g.DrawString(fTitle, font, brush,
                    (width - w) / 2f, h);

                string xTitle = "Krok czasowy";
                w = g.MeasureString(xTitle, font).Width;
                g.DrawString(xTitle, font, brush,
                    sx0 + (sx1 - sx0 - w) / 2f, sy1 + h + h);

                string yTitle = title;
                w = g.MeasureString(yTitle, font).Width;
                g.DrawString(yTitle, font, brush,
                    sx1 + w / 5f, sy0 + (sy1 - sy0) / 2f - h / 2f);

                int i = 0;

                while (i <= 8)
                {
                    float sx = xSlope * x + xInter;
                    string s = string.Format("{0,5:0.00}", Math.Log10(x + 1));

                    g.DrawLine(pen1, sx, sy0, sx, sy1);
                    w = g.MeasureString(s, font).Width;
                    g.DrawString(s, font, brush,
                        sx - w / 2, sy1 + h / 2f);
                    x += deltaX;
                    i++;
                }

                i = 0;

                while (i <= 8)
                {
                    float sy = ySlope * y + yInter;
                    string s = string.Format("{0,5:0.00}", y);

                    w = g.MeasureString(s, font).Width;
                    g.DrawLine(pen1, sx0, sy, sx1, sy);
                    g.DrawString(s, font, brush,
                        sx0 - w - w / 5f, sy - h / 2f);
                    y += deltaY;
                    i++;
                }

                g.Clip = new Region(new RectangleF(
                    sx0, sy0, (sx1 - sx0), (sy1 - sy0)));

                for (i = 0; i < n; i++)
                {
                    float px = (float)pts[i].x;
                    float py = (float)pts[i].y;

                    float sx = xSlope * px + xInter;
                    float sy = ySlope * py + yInter;

                    g.FillEllipse(potBrush, (float)sx, (float)sy, (float)dot, (float)dot);
                }
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.ToString(), "Warning Message",
                    MessageBoxButtons.OK, MessageBoxIcon.Warning);
            }
        }

        private void LayOutTheForm()
        {

            int w = ClientSize.Width;
            int h = ClientSize.Height;

            panel1.Width = w;
            panel1.Height = h;
            panel1.Location = new Point(0, 0);
            panel1.Invalidate();
        }

        protected void PanelPaintHandler(object sender, PaintEventArgs pa)
        {
            DrawGraph((float)xMin, (float)yMin, (float)xMax, (float)yMax, pa.Graphics);
        }

        protected override void OnResize(EventArgs ea)
        {
            LayOutTheForm();
        }
    }
}
