#pragma once

#ifndef SolarSystem_H
#define SolarSystem_H

#include "PunktMaterialny.h"

#define _USE_MATH_DEFINES
#include <math.h>
#include <time.h>
#include<fstream>

class Planeta {

public:
	double skala_odleglosci = 1.0;
	double KM_na_AU = 0.0000000066845;
	double odlegloscOdSlonca_AU;
	double rednicaRwnikowa_AU;
	double masa;
	double czasObieguSoca_s;
	double okresObrotu_s;
	double OdlegoOdSoca() { return skala_odleglosci*odlegloscOdSlonca_AU; }

	Planeta(double odlego_od_Soca_KM, double rednicaRwnikowa_KM, double ile_mas_ziemi, double czasObiegu_d,double okresObrotu_h = 0) {

		odlegloscOdSlonca_AU = KM_na_AU * odlego_od_Soca_KM;
		rednicaRwnikowa_AU = KM_na_AU * rednicaRwnikowa_KM;
		masa = ile_mas_ziemi;
		czasObieguSoca_s = czasObiegu_d * 24 * 3600;
		okresObrotu_s = okresObrotu_h * 3600;
	}
	double prdkoKtowa() {

		return 2 * M_PI / czasObieguSoca_s;
	}
	double PromiePlanety() { return skala_odleglosci*rednicaRwnikowa_AU / 2; }
};

class SolarSystem : public ZbiorPunktowMaterialnych {

private:
	double G;
	Wektor pooenieSoca = Wektor((polozenieMax.X - polozenieMin.X) / 2, (polozenieMax.Y - polozenieMin.Y) / 2, (polozenieMax.Z - polozenieMin.Z) / 2);

	const double predkosci[10][8][3] = {
		{{ -3.372440409499573E-02 ,2.764221478256139E-03,3.319756298599656E-03 }, { 1.171575332462580E-02 ,1.631734737116074E-02 ,-4.522498211961549E-04 },{ 1.692976221225897E-02 ,-1.925781753922310E-04 ,2.087146512288498E-08 },{ -1.281926732559828E-02 ,-3.007251523859464E-03 ,2.515892564619911E-04 },{ 2.933984503882877E-03 ,-6.563422366929296E-03 ,-3.834621794315080E-05 },{ 5.250691366147950E-03 ,-5.635809468187713E-04 ,-1.995167189250283E-04 },{ -1.684942909463664E-03 ,3.375937763866848E-03 ,3.432377771556653E-05 },{ 9.458303189767470E-04 ,3.000624584805501E-03 ,-8.405865622714694E-05 } },
		{{ -3.374564593937646E-02 ,-3.283882563264605E-04,3.069000378190143E-03 }, { 1.125946486568502E-02 ,1.663737189307506E-02 ,-4.215293059832505E-04 }, { 1.693054952544072E-02 ,9.541247446833135E-05 ,-1.574323553378871E-07 }, { -1.278460125515715E-02 ,-3.115945815690588E-03 ,2.484606933479410E-04 }, { 2.943018801125628E-03 ,-6.559934189540601E-03 ,-3.858603658446232E-05 }, { 5.250746507186180E-03 ,-5.607207092221752E-04 ,-1.995249015038772E-04 }, { -1.685544443496479E-03 ,3.375618495045974E-03 ,3.436571646351921E-05 }, { 9.451093782312960E-04 ,3.000385976729779E-03 ,-8.404729819081821E-05 }},
		{{ -3.343274689768666E-02 ,-3.371774853004209E-03,2.791612425967061E-03 },{ 1.079447957098730E-02 ,1.694468559917884E-02 ,-3.904812812946081E-04 },{ 1.692615034570773E-02,3.835855351697323E-04 ,-3.341708967849293E-07 },{ -1.274909360480118E-02 ,-3.224222303658399E-03,2.453202311790624E-04 },{ 2.950873200712548E-03 ,-6.556366358090907E-03 ,-3.884281763505195E-05 },{ 5.250796987058316E-03 ,-5.579601495020711E-04 ,-1.994814958034608E-04 },{ -1.686235885506023E-03 ,3.375288647264497E-03 ,3.419236725123469E-05 },{ 9.444143860451515E-04 ,3.000560127180463E-03 ,-8.367388808151601E-05 }},
		{{ -3.280226005127379E-02 ,-6.320319601370305E-03,2.492838641290788E-03 },{ 1.032114441114871E-02 ,1.723905369683545E-02 ,-3.591289896960656E-04 },{ 1.691649951620368E-02 ,6.717338679447024E-04 ,-4.961120317895905E-07 },{ -1.271274954127306E-02 ,-3.332076056189247E-03,2.421681041895970E-04 },{ 2.961213814940795E-03 ,-6.552681440656707E-03 ,-3.905702872449362E-05 },{ 5.250962015784669E-03 ,-5.552534752574964E-04 ,-1.994218794527100E-04 },{ -1.687096370369844E-03 ,3.375023681087059E-03 ,3.421683151967972E-05 },{ 9.441484651035647E-04 ,3.001077230504509E-03 ,-8.332097454064757E-05 }},
		{{ -3.187811774384755E-02 ,-9.133875569515055E-03,2.178155086666944E-03 },{ 9.839812603541687E-03 ,1.752025059171514E-02 ,-3.274959080730911E-04 },{ 1.690158416973732E-02 ,9.596316655789683E-04 ,-6.312277728550816E-07 },{ -1.267557422250766E-02 ,-3.439502199036299E-03 ,2.390045418328370E-04 },{ 2.969788443315776E-03 ,-6.549163764232092E-03 ,-3.930193133395919E-05 },{ 5.251191642215092E-03 ,-5.526174406953598E-04 ,-1.993321460034910E-04 },{ -1.687771674549766E-03 ,3.374728838883452E-03,3.431771063649805E-05 },{ 9.442692456769184E-04 ,3.001511365963167E-03 ,-8.334982585324417E-05 }},
		{{ -3.068985337403566E-02 ,-1.177900030067648E-02,1.853003827924408E-03 },{ 9.350843387119119E-03 , 1.778806004970218E-02 ,-2.956057324404599E-04 },{ 1.688144699850170E-02 ,1.247059342716200E-03 ,-7.307168388328889E-07 },{ -1.263757279509735E-02 ,-3.546495910780734E-03 ,2.358297681949739E-04 },{ 2.979579053586421E-03 ,-6.544488972258932E-03 ,-3.949221058961264E-05 },{ 5.251611292510389E-03 ,-5.499845442758835E-04 ,-1.992593690075412E-04 },{ -1.688495818506236E-03 ,3.374440147411830E-03,3.438977585261971E-05 },{ 9.443328634629772E-04 ,3.001522723943240E-03 ,-8.372532896529828E-05 }},
		{{ -2.927068266582542E-02 ,-1.422958276459683E-02,1.522565679771416E-03 },{ 8.854601793282789E-03 ,1.804227535434295E-02 ,-2.634823625789585E-04 },{ 1.685617558674296E-02 ,1.533825722534595E-03 ,-7.901164328464014E-07 },{ -1.259875039884981E-02 ,-3.653052423372090E-03 ,2.326440043388055E-04 },{ 2.988923749828380E-03 ,-6.541254194815304E-03 ,-3.973835826815390E-05 },{ 5.252128862136224E-03 ,-5.472568043773730E-04 ,-1.992452088680509E-04 },{ -1.689081373870715E-03 ,3.374128959158125E-03 ,3.451795147867315E-05 },{ 9.439548207355212E-04 ,3.001210411171206E-03 ,-8.405246768946703E-05 }},
		{{ -2.765564402429583E-02 ,-1.646690630085870E-02,1.191584458310082E-03 },{ 8.351458412795502E-03 ,1.828269945974404E-02 ,-2.311498864995304E-04 },{ 1.682588320895744E-02 ,1.819779920542428E-03 ,-8.092127308326997E-07 },{ -1.255911217603784E-02,-3.759167021194631E-03 ,2.294474730420499E-04 },{ 2.997901330664399E-03 ,-6.535721405880127E-03 ,-3.990448303104883E-05 },{ 5.252670353750630E-03 ,-5.444243510427804E-04 ,-1.992881712249237E-04 },{ -1.689688521493219E-03 , 3.373796911930530E-03 ,3.445957169760054E-05 },{ 9.432091397941582E-04 ,3.001009542176200E-03 ,-8.398643663884427E-05 }},
		{{ -2.587995065739099E-02 ,-1.847924850054179E-02,8.642484989104103E-04 },{ 7.841789158742575E-03 ,1.850914513881686E-02 ,-1.986325647365403E-04 },{ 1.679068938897593E-02 ,2.104811444609906E-03 ,-7.910618330812996E-07 },{ -1.251866327716391E-02 ,-3.864835041556402E-03 ,2.262404018497476E-04 },{ 3.006938541623970E-03 ,-6.532180187683707E-03  ,-4.014412880583398E-05 },{ 5.253251809445082E-03 ,-5.415466095583516E-04 ,-1.993575653860017E-04 },{ -1.690380683080911E-03 ,3.373504367093711E-03  ,3.455658082273054E-05 },{ 9.425515342793195E-04 ,3.001239527775064E-03 ,-8.359063138207106E-05 }},
		{{ -2.397764847356956E-02 ,-2.026114915518181E-02,5.441273490637328E-04 },{ 7.325975025559134E-03 , 1.872143512675346E-02  ,-1.659548144735084E-04 },{ 1.675070614310727E-02 ,2.388842367815096E-03 ,-7.407092126649716E-07 },{ -1.247740885526992E-02 ,-3.970051881193017E-03 ,2.230230201640951E-04 },{ 3.014928925536883E-03 ,-6.528147836173221E-03 ,-4.038504609961809E-05 },{ 5.253816812096724E-03 ,-5.385149683120611E-04 ,-1.995054216179218E-04 },{ -1.690868120751228E-03 ,3.373153994428583E-03 ,3.456269965877614E-05 },{ 9.423462536398096E-04 ,3.001772417445146E-03 ,-8.327085175061328E-05 }}
	
	};
public:
	Wektor polozenieMin, polozenieMax;
	const double polozenia[32][8][3] = { 

		{ { 1.895852835847268E-02 ,3.067030918973734E-01 ,2.332218678050880E-02 },{ 5.893951455206983E-01 ,-4.256481619192217E-01 ,-3.985042133901953E-02 },{ -7.300343959802882E-03 ,-1.016225434907809E+00 ,4.478009409438041E-05 },{ -4.826601790492047E-01 ,1.535014161608938E+00 ,4.401237182772826E-02 },{ -4.995167224343685E+00 ,-2.183256724173861E+00 ,1.208393270491884E-01 },{ -9.761485701716526E-01 ,-1.000872765724515E+01 ,2.128173694007556E-01 },{ 1.806089214512000E+01 ,8.407925584588700E+00 ,-2.026288799236440E-01 } ,{ 2.850219591706601E+01 ,-9.181882704654404E+00 ,-4.678035022457146E-01 }},
		{ { -1.480480788625003E-02 ,3.079189202766085E-01 ,2.651899176396551E-02 },{ 6.008834936402653E-01 ,-4.091697530751141E-01 ,-4.028733915704000E-02 },{ 9.630240278067135E-03 ,-1.016274040704775E+00 ,4.471222400415887E-05 },{ -4.954621836835322E-01 ,1.531952528349504E+00 ,4.426239780414443E-02 },{ -4.992228932260023E+00 ,-2.189818599495410E+00 ,1.208008514445308E-01 },{ -9.708978507837706E-01 ,-1.000928980601700E+01 ,2.126178476444577E-01 },{ 1.805920688213446E+01 ,8.411301368733071E+00 ,-2.025945207405735E-01 } ,{ 2.850314140650500E+01 ,-9.178882223552026E+00 ,-4.678875930772843E-01 }},
		{ { -4.842128472061506E-02 ,3.060627645116081E-01 ,2.945130454019978E-02 },{ 6.119111762349444E-01 ,-3.923776551720365E-01 ,-4.069337077839014E-02 },{ 2.655902584569833E-02 ,-1.016034548728227E+00 ,4.446573965506483E-05 },{ -5.082291010405413E-01 ,1.528782409278041E+00 ,4.450928925422038E-02 },{ -4.989281966322111E+00 ,-2.196376383900251E+00 ,1.207621502347684E-01 },{ -9.656470873387945E-01 ,-1.000984913257848E+01 ,2.124183379801596E-01 },{ 1.805752102022262E+01 ,8.414676816521091E+00 ,-2.025602391131248E-01 } ,{ 2.850408614481556E+01 ,-9.175881789649070E+00 ,-4.679714713641187E-01 }},
		{ { -8.156435660840312E-02 ,3.012070460839972E-01 ,3.209508546965226E-02 },{ 6.224696694244458E-01 ,-3.752846971817812E-01 ,-4.106820029472213E-02 },{ 4.348078982093539E-02 ,-1.015506877659796E+00 ,4.404882822311114E-05 },{ -5.209600921175077E-01 ,1.525504224662848E+00 , 4.475303439265241E-02 },{ -4.986325898848297E+00 ,-2.202931311109467E+00 ,1.207231860066446E-01 },{ -9.603962111692264E-01 ,-1.001040573980665E+01 ,2.122188870321093E-01 },{ 1.805583434797467E+01 ,8.418051970666273E+00 ,-2.025260602241492E-01 } ,{ 2.850503038398139E+01 ,-9.172880984438970E+00 ,-4.680549479568122E-01 }},
		{ { -1.139278572541411E-01 ,2.934671968816662E-01 ,3.443167899411696E-02 },{ 6.325507994410740E-01 ,-3.579039382832995E-01 ,-4.141153515867245E-02 },{ 6.039026940168915E-02 ,-1.014691164577196E+00 ,4.348249456395913E-05 },{ -5.336543230683639E-01 ,1.522118399684967E+00, 4.499362166401532E-02 },{ -4.983360556202117E+00 ,-2.209481944796025E+00  ,1.206840148306782E-01 },{ -9.551451481564704E-01 ,-1.001095966704131E+01 ,2.120195069844349E-01 },{ 1.805414690500360E+01 ,8.421426850784195E+00 ,-2.024917781845656E-01 } ,{ 2.850597457576565E+01 ,-9.169879663941856E+00 ,-4.681382456222194E-01 }},
		{ { -1.452325105821608E-01 ,2.829955209091589E-01 ,3.644790938112356E-02 },{ 6.421467487642250E-01 ,-3.402486585830730E-01 ,-4.172310641060083E-02 },{ 7.728221704144093E-02 ,-1.013587771366220E+00 ,4.279829817133926E-05 },{ -5.463109652024464E-01 ,1.518625364378343E+00 ,4.523103974148721E-02 },{ -4.980385597355820E+00 ,-2.216028962771276E+00,1.206446150343594E-01 },{ -9.498937607983444E-01 ,-1.001151097579472E+01 ,2.118202167479250E-01 },{ 1.805245877239870E+01,8.424801432971840E+00 ,-2.024574384157649E-01 } ,{ 2.850691890266074E+01 ,-9.166878108230108E+00 ,-4.682217677482369E-01 }},
		{ { -1.752305752632490E-01 ,2.699741264809217E-01 ,3.813592928641262E-02 },{ 6.512500620277317E-01 ,-3.223323496758007E-01 ,-4.200266889005212E-02 },{ 9.415145170005551E-02 ,-1.012197266788682E+00 ,4.203447248800790E-05 },{ -5.589291949829684E-01 ,1.515025553571849E+00 ,4.546527752587818E-02 },{ -4.977401701900749E+00 ,-2.222571809733576E+00,1.206049951749341E-01 },{ -9.446418973437306E-01 ,-1.001205960391376E+01 , 2.116209688795682E-01 },{ 1.805076996840996E+01 ,8.428175722508254E+00 ,-2.024229745426553E-01 } ,{ 2.850786308842588E+01 ,-9.163876730668568E+00 ,-4.683056795184251E-01 }},
		{ { -2.037085850767302E-01 ,2.546075198426070E-01 ,3.949286596234194E-02 },{ 6.598536516960981E-01 ,-3.041687050499045E-01 ,-4.225000143176892E-02 },{ 1.109928944986940E-01 ,-1.010520391107965E+00 ,4.123153643547587E-05 },{ -5.715081940241061E-01 ,1.511319406832763E+00 ,4.569632414441168E-02 },{ -4.974407879198715E+00 ,-2.229110239741156E+00 ,1.205651815853473E-01 },{ -9.393894939444503E-01 ,-1.001260545089306E+01 ,2.114217052057965E-01 },{ 1.804908060497397E+01 ,8.431549681906365E+00 ,-2.023884775188359E-01 } ,{ 2.850880668488841E+01 ,-9.160875648785110E+00 ,-4.683897364219684E-01 }},
		{ { -2.304883251699494E-01 ,2.371153774954020E-01, 4.052032022933257E-02 },{ 6.679508035056695E-01 ,-2.857716103424539E-01 ,-4.246490704608254E-02 },{ 1.278015843975006E-01 ,-1.008558014870399E+00 ,4.042848456814586E-05 },{ -5.840471490868364E-01 ,1.507507368410230E+00 ,4.592416894917158E-02 },{ -4.971405721521898E+00 ,-2.235644326816073E+00 ,1.205251492385576E-01 },{ -9.341365422977397E-01 ,-1.001314844646605E+01 ,2.112223883841182E-01 },{ 1.804739055179396E+01 ,8.434923333979913E+00  ,-2.023539870202989E-01 } ,{ 2.850974953723031E+01 ,-9.157874562261775E+00 ,-4.684735381010792E-01 }},
		{ { -2.554263079714484E-01 ,2.177258960427159E-01 ,4.122377499591562E-02 },{ 6.755351816667960E-01 ,-2.671551334482213E-01 ,-4.264721308351276E-02 },{ 1.445726769651739E-01 ,-1.006311101996554E+00 ,3.966015594530124E-05 },{ -5.965452520746580E-01 ,1.503589887178167E+00 ,4.614880151566404E-02 },{ -4.968394672363556E+00 ,-2.242174155923466E+00 ,1.204848982142525E-01 },{ -9.288829976793662E-01 ,-1.001368848677512E+01 ,2.110229608117581E-01 },{ 1.804569992479703E+01 ,8.438296667374070E+00 ,-2.023194018851640E-01 } ,{ 2.851069194472082E+01 ,-9.154873064751262E+00 ,-4.685569440334691E-01 }},
		{ { -2.784120775664587E-01 ,1.966699834852131E-01 ,4.161196505839607E-02 },{ 6.826008338230675E-01 ,-2.483335144876966E-01 ,-4.279677138341229E-02 },{ -7.300343959802882E-03 ,-1.016225434907809E+00 ,4.478009409438041E-05 },{ -4.826601790492047E-01 ,1.535014161608938E+00 ,4.401237182772826E-02 },{ -4.995167224343685E+00 ,-2.183256724173861E+00 ,1.208393270491884E-01 },{ -9.761485701716526E-01 ,-1.000872765724515E+01 ,2.128173694007556E-01 },{ 1.806089214512000E+01 ,8.407925584588700E+00 ,-2.026288799236440E-01 } ,{ 2.850219591706601E+01 ,-9.181882704654404E+00 ,-4.678035022457146E-01 }},
		{ { -2.993656847648496E-01 ,1.741764277191109E-01 ,4.169625066927631E-02 },{ 5.893951455206983E-01 ,-4.256481619192217E-01 ,-3.985042133901953E-02 },{ -7.300343959802882E-03 ,-1.016225434907809E+00 ,4.478009409438041E-05 },{ -4.826601790492047E-01 ,1.535014161608938E+00 ,4.401237182772826E-02 },{ -4.995167224343685E+00 ,-2.183256724173861E+00 ,1.208393270491884E-01 },{ -9.761485701716526E-01 ,-1.000872765724515E+01 ,2.128173694007556E-01 },{ 1.806089214512000E+01 ,8.407925584588700E+00 ,-2.026288799236440E-01 } ,{ 2.850219591706601E+01 ,-9.181882704654404E+00 ,-4.678035022457146E-01 }},
		{ { -3.182346503192922E-01 ,1.504680714228550E-01 ,4.149002638287932E-02 },{ 5.893951455206983E-01 ,-4.256481619192217E-01 ,-3.985042133901953E-02 },{ -7.300343959802882E-03 ,-1.016225434907809E+00 ,4.478009409438041E-05 },{ -4.826601790492047E-01 ,1.535014161608938E+00 ,4.401237182772826E-02 },{ -4.995167224343685E+00 ,-2.183256724173861E+00 ,1.208393270491884E-01 },{ -9.761485701716526E-01 ,-1.000872765724515E+01 ,2.128173694007556E-01 },{ 1.806089214512000E+01 ,8.407925584588700E+00 ,-2.026288799236440E-01 } ,{ 2.850219591706601E+01 ,-9.181882704654404E+00 ,-4.678035022457146E-01 }},
		{ { -3.349906858148964E-01 ,1.257589440991329E-01 ,4.100818588116992E-02 },{ 5.893951455206983E-01 ,-4.256481619192217E-01 ,-3.985042133901953E-02 },{ -7.300343959802882E-03 ,-1.016225434907809E+00 ,4.478009409438041E-05 },{ -4.826601790492047E-01 ,1.535014161608938E+00 ,4.401237182772826E-02 },{ -4.995167224343685E+00 ,-2.183256724173861E+00 ,1.208393270491884E-01 },{ -9.761485701716526E-01 ,-1.000872765724515E+01 ,2.128173694007556E-01 },{ 1.806089214512000E+01 ,8.407925584588700E+00 ,-2.026288799236440E-01 } ,{ 2.850219591706601E+01 ,-9.181882704654404E+00 ,-4.678035022457146E-01 }},
		{ { -3.496263839127759E-01 ,1.002522518291323E-01 ,4.026665407383876E-02 },{ 5.893951455206983E-01 ,-4.256481619192217E-01 ,-3.985042133901953E-02 },{ -7.300343959802882E-03 ,-1.016225434907809E+00 ,4.478009409438041E-05 },{ -4.826601790492047E-01 ,1.535014161608938E+00 ,4.401237182772826E-02 },{ -4.995167224343685E+00 ,-2.183256724173861E+00 ,1.208393270491884E-01 },{ -9.761485701716526E-01 ,-1.000872765724515E+01 ,2.128173694007556E-01 },{ 1.806089214512000E+01 ,8.407925584588700E+00 ,-2.026288799236440E-01 } ,{ 2.850219591706601E+01 ,-9.181882704654404E+00 ,-4.678035022457146E-01 }},
		{ { -3.621520317393091E-01 ,7.413909933164277E-02 ,3.928199033420330E-02 },{ 5.893951455206983E-01 ,-4.256481619192217E-01 ,-3.985042133901953E-02 },{ -7.300343959802882E-03 ,-1.016225434907809E+00 ,4.478009409438041E-05 },{ -4.826601790492047E-01 ,1.535014161608938E+00 ,4.401237182772826E-02 },{ -4.995167224343685E+00 ,-2.183256724173861E+00 ,1.208393270491884E-01 },{ -9.761485701716526E-01 ,-1.000872765724515E+01 ,2.128173694007556E-01 },{ 1.806089214512000E+01 ,8.407925584588700E+00 ,-2.026288799236440E-01 } ,{ 2.850219591706601E+01 ,-9.181882704654404E+00 ,-4.678035022457146E-01 }},
		{ { -3.725926494314331E-01 ,4.759781147155022E-02 ,3.807106137343860E-02 },{ 5.893951455206983E-01 ,-4.256481619192217E-01 ,-3.985042133901953E-02 },{ -7.300343959802882E-03 ,-1.016225434907809E+00 ,4.478009409438041E-05 },{ -4.826601790492047E-01 ,1.535014161608938E+00 ,4.401237182772826E-02 },{ -4.995167224343685E+00 ,-2.183256724173861E+00 ,1.208393270491884E-01 },{ -9.761485701716526E-01 ,-1.000872765724515E+01 ,2.128173694007556E-01 },{ 1.806089214512000E+01 ,8.407925584588700E+00 ,-2.026288799236440E-01 } ,{ 2.850219591706601E+01 ,-9.181882704654404E+00 ,-4.678035022457146E-01 }},
		{ { -3.809853131900864E-01 ,2.079372655465838E-02 ,3.665077876645811E-02 },{ 5.893951455206983E-01 ,-4.256481619192217E-01 ,-3.985042133901953E-02 },{ -7.300343959802882E-03 ,-1.016225434907809E+00 ,4.478009409438041E-05 },{ -4.826601790492047E-01 ,1.535014161608938E+00 ,4.401237182772826E-02 },{ -4.995167224343685E+00 ,-2.183256724173861E+00 ,1.208393270491884E-01 },{ -9.761485701716526E-01 ,-1.000872765724515E+01 ,2.128173694007556E-01 },{ 1.806089214512000E+01 ,8.407925584588700E+00 ,-2.026288799236440E-01 } ,{ 2.850219591706601E+01 ,-9.181882704654404E+00 ,-4.678035022457146E-01 }},
		{ { -3.873767893936157E-01 ,-6.120653511487715E-03 ,3.503789417489657E-02 },{ 5.893951455206983E-01 ,-4.256481619192217E-01 ,-3.985042133901953E-02 },{ -7.300343959802882E-03 ,-1.016225434907809E+00 ,4.478009409438041E-05 },{ -4.826601790492047E-01 ,1.535014161608938E+00 ,4.401237182772826E-02 },{ -4.995167224343685E+00 ,-2.183256724173861E+00 ,1.208393270491884E-01 },{ -9.761485701716526E-01 ,-1.000872765724515E+01 ,2.128173694007556E-01 },{ 1.806089214512000E+01 ,8.407925584588700E+00 ,-2.026288799236440E-01 } ,{ 2.850219591706601E+01 ,-9.181882704654404E+00 ,-4.678035022457146E-01 }},
		{ { -3.918214827014898E-01 ,-3.300525474143379E-02 ,3.324884448321288E-02 },{ 5.893951455206983E-01 ,-4.256481619192217E-01 ,-3.985042133901953E-02 },{ -7.300343959802882E-03 ,-1.016225434907809E+00 ,4.478009409438041E-05 },{ -4.826601790492047E-01 ,1.535014161608938E+00 ,4.401237182772826E-02 },{ -4.995167224343685E+00 ,-2.183256724173861E+00 ,1.208393270491884E-01 },{ -9.761485701716526E-01 ,-1.000872765724515E+01 ,2.128173694007556E-01 },{ 1.806089214512000E+01 ,8.407925584588700E+00 ,-2.026288799236440E-01 } ,{ 2.850219591706601E+01 ,-9.181882704654404E+00 ,-4.678035022457146E-01 }},
		{ { -3.943796852415308E-01 ,-5.973184557186780E-02 ,3.129963902130992E-02 },{ 5.893951455206983E-01 ,-4.256481619192217E-01 ,-3.985042133901953E-02 },{ -7.300343959802882E-03 ,-1.016225434907809E+00 ,4.478009409438041E-05 },{ -4.826601790492047E-01 ,1.535014161608938E+00 ,4.401237182772826E-02 },{ -4.995167224343685E+00 ,-2.183256724173861E+00 ,1.208393270491884E-01 },{ -9.761485701716526E-01 ,-1.000872765724515E+01 ,2.128173694007556E-01 },{ 1.806089214512000E+01 ,8.407925584588700E+00 ,-2.026288799236440E-01 } ,{ 2.850219591706601E+01 ,-9.181882704654404E+00 ,-4.678035022457146E-01 }},
		{ { -3.951161042969719E-01 ,-8.618333920223618E-02 ,2.920578150161535E-02 },{ 5.893951455206983E-01 ,-4.256481619192217E-01 ,-3.985042133901953E-02 },{ -7.300343959802882E-03 ,-1.016225434907809E+00 ,4.478009409438041E-05 },{ -4.826601790492047E-01 ,1.535014161608938E+00 ,4.401237182772826E-02 },{ -4.995167224343685E+00 ,-2.183256724173861E+00 ,1.208393270491884E-01 },{ -9.761485701716526E-01 ,-1.000872765724515E+01 ,2.128173694007556E-01 },{ 1.806089214512000E+01 ,8.407925584588700E+00 ,-2.026288799236440E-01 } ,{ 2.850219591706601E+01 ,-9.181882704654404E+00 ,-4.678035022457146E-01 }},
		{ { -3.940986408329783E-01 ,-1.122530339879023E-01 ,2.698222003011279E-02 },{ 5.893951455206983E-01 ,-4.256481619192217E-01 ,-3.985042133901953E-02 },{ -7.300343959802882E-03 ,-1.016225434907809E+00 ,4.478009409438041E-05 },{ -4.826601790492047E-01 ,1.535014161608938E+00 ,4.401237182772826E-02 },{ -4.995167224343685E+00 ,-2.183256724173861E+00 ,1.208393270491884E-01 },{ -9.761485701716526E-01 ,-1.000872765724515E+01 ,2.128173694007556E-01 },{ 1.806089214512000E+01 ,8.407925584588700E+00 ,-2.026288799236440E-01 } ,{ 2.850219591706601E+01 ,-9.181882704654404E+00 ,-4.678035022457146E-01 }},
		{ { -3.913973893781951E-01,-1.378438296183277E-01 ,2.464331940320102E-02 },{ 5.893951455206983E-01 ,-4.256481619192217E-01 ,-3.985042133901953E-02 },{ -7.300343959802882E-03 ,-1.016225434907809E+00 ,4.478009409438041E-05 },{ -4.826601790492047E-01 ,1.535014161608938E+00 ,4.401237182772826E-02 },{ -4.995167224343685E+00 ,-2.183256724173861E+00 ,1.208393270491884E-01 },{ -9.761485701716526E-01 ,-1.000872765724515E+01 ,2.128173694007556E-01 },{ 1.806089214512000E+01 ,8.407925584588700E+00 ,-2.026288799236440E-01 } ,{ 2.850219591706601E+01 ,-9.181882704654404E+00 ,-4.678035022457146E-01 }},
		{ { -3.870838301217943E-01 ,-1.628674465545730E-01 ,2.220285077212263E-02 },{ 5.893951455206983E-01 ,-4.256481619192217E-01 ,-3.985042133901953E-02 },{ -7.300343959802882E-03 ,-1.016225434907809E+00 ,4.478009409438041E-05 },{ -4.826601790492047E-01 ,1.535014161608938E+00 ,4.401237182772826E-02 },{ -4.995167224343685E+00 ,-2.183256724173861E+00 ,1.208393270491884E-01 },{ -9.761485701716526E-01 ,-1.000872765724515E+01 ,2.128173694007556E-01 },{ 1.806089214512000E+01 ,8.407925584588700E+00 ,-2.026288799236440E-01 } ,{ 2.850219591706601E+01 ,-9.181882704654404E+00 ,-4.678035022457146E-01 }},
		{ { -3.812301857759232E-01 ,-1.872436680098009E-01 ,1.967399458096716E-02 },{ 5.893951455206983E-01 ,-4.256481619192217E-01 ,-3.985042133901953E-02 },{ -7.300343959802882E-03 ,-1.016225434907809E+00 ,4.478009409438041E-05 },{ -4.826601790492047E-01 ,1.535014161608938E+00 ,4.401237182772826E-02 },{ -4.995167224343685E+00 ,-2.183256724173861E+00 ,1.208393270491884E-01 },{ -9.761485701716526E-01 ,-1.000872765724515E+01 ,2.128173694007556E-01 },{ 1.806089214512000E+01 ,8.407925584588700E+00 ,-2.026288799236440E-01 } ,{ 2.850219591706601E+01 ,-9.181882704654404E+00 ,-4.678035022457146E-01 }},
		{ { -3.739089181892133E-01 ,-2.108996173800941E-01 ,1.706935342873865E-02 },{ 5.893951455206983E-01 ,-4.256481619192217E-01 ,-3.985042133901953E-02 },{ -7.300343959802882E-03 ,-1.016225434907809E+00 ,4.478009409438041E-05 },{ -4.826601790492047E-01 ,1.535014161608938E+00 ,4.401237182772826E-02 },{ -4.995167224343685E+00 ,-2.183256724173861E+00 ,1.208393270491884E-01 },{ -9.761485701716526E-01 ,-1.000872765724515E+01 ,2.128173694007556E-01 },{ 1.806089214512000E+01 ,8.407925584588700E+00 ,-2.026288799236440E-01 } ,{ 2.850219591706601E+01 ,-9.181882704654404E+00 ,-4.678035022457146E-01 }},
		{ { -3.651923424643270E-01 ,-2.337690791840979E-01 ,1.440097215606081E-02 },{ 5.893951455206983E-01 ,-4.256481619192217E-01 ,-3.985042133901953E-02 },{ -7.300343959802882E-03 ,-1.016225434907809E+00 ,4.478009409438041E-05 },{ -4.826601790492047E-01 ,1.535014161608938E+00 ,4.401237182772826E-02 },{ -4.995167224343685E+00 ,-2.183256724173861E+00 ,1.208393270491884E-01 },{ -9.761485701716526E-01 ,-1.000872765724515E+01 ,2.128173694007556E-01 },{ 1.806089214512000E+01 ,8.407925584588700E+00 ,-2.026288799236440E-01 } ,{ 2.850219591706601E+01 ,-9.181882704654404E+00 ,-4.678035022457146E-01 }},
		{ { -3.551523391576401E-01 ,-2.557918679680620E-01 ,1.168036301059099E-02 },{ 5.893951455206983E-01 ,-4.256481619192217E-01 ,-3.985042133901953E-02 },{ -7.300343959802882E-03 ,-1.016225434907809E+00 ,4.478009409438041E-05 },{ -4.826601790492047E-01 ,1.535014161608938E+00 ,4.401237182772826E-02 },{ -4.995167224343685E+00 ,-2.183256724173861E+00 ,1.208393270491884E-01 },{ -9.761485701716526E-01 ,-1.000872765724515E+01 ,2.128173694007556E-01 },{ 1.806089214512000E+01 ,8.407925584588700E+00 ,-2.026288799236440E-01 } ,{ 2.850219591706601E+01 ,-9.181882704654404E+00 ,-4.678035022457146E-01 }},
		{ { -3.438601478519633E-01 ,-2.769132470238289E-01 ,8.918534207280610E-03 },{ 5.893951455206983E-01 ,-4.256481619192217E-01 ,-3.985042133901953E-02 },{ -7.300343959802882E-03 ,-1.016225434907809E+00 ,4.478009409438041E-05 },{ -4.826601790492047E-01 ,1.535014161608938E+00 ,4.401237182772826E-02 },{ -4.995167224343685E+00 ,-2.183256724173861E+00 ,1.208393270491884E-01 },{ -9.761485701716526E-01 ,-1.000872765724515E+01 ,2.128173694007556E-01 },{ 1.806089214512000E+01 ,8.407925584588700E+00 ,-2.026288799236440E-01 } ,{ 2.850219591706601E+01 ,-9.181882704654404E+00 ,-4.678035022457146E-01 }},
		{ { -3.313862278970339E-01 ,-2.970833969328362E-01 ,6.126020579217454E-03 },{ 5.893951455206983E-01 ,-4.256481619192217E-01 ,-3.985042133901953E-02 },{ -7.300343959802882E-03 ,-1.016225434907809E+00 ,4.478009409438041E-05 },{ -4.826601790492047E-01 ,1.535014161608938E+00 ,4.401237182772826E-02 },{ -4.995167224343685E+00 ,-2.183256724173861E+00 ,1.208393270491884E-01 },{ -9.761485701716526E-01 ,-1.000872765724515E+01 ,2.128173694007556E-01 },{ 1.806089214512000E+01 ,8.407925584588700E+00 ,-2.026288799236440E-01 } ,{ 2.850219591706601E+01 ,-9.181882704654404E+00 ,-4.678035022457146E-01 }}
	};
	
	double predkosc_poczatkowa_pocisku = 2.48;
	double v_pocisku;
	 double SumaPromieni;
	 double odleglosc_Neptuna;
	Planeta** tablicaPlanet;
	bool flaga_pocisk = false;
	

	SolarSystem(int liczbaPlanet, double stalaGrawitacji, Wektor polozenieMin, Wektor polozenieMax)
		:ZbiorPunktowMaterialnych(liczbaPlanet), G(stalaGrawitacji) {

		tablicaPlanet = new Planeta*[liczbaPlanet];
		tablicaPlanet[0] = new Planeta(0, 1392000, 332950, 0);
		tablicaPlanet[1] = new Planeta(57909170, 4879, 0.0552, 87969);
		tablicaPlanet[2] = new Planeta(108208926, 12104, 0.8149, 224, 701);
		tablicaPlanet[3] = new Planeta(152141051.079, 10, 1.0, 365.256);//12756
		tablicaPlanet[4] = new Planeta(227936637, 6805, 0.1074, 686.960);
		tablicaPlanet[5] = new Planeta(778412027, 142984, 317.8, 4333.287);
		tablicaPlanet[6] = new Planeta(1426725413, 120536, 95.1620, 10756.2);
		tablicaPlanet[7] = new Planeta(2870972220, 51118, 14.5360, 30707.490);
		tablicaPlanet[8] = new Planeta(4498252900, 49528, 17.1470, 60223.353);
		tablicaPlanet[9] = new Planeta(152141051.08, 127560, 1e-24, 0, 0);
		SumaPromieni = suma_promieni_planet();
		v_pocisku = predkosc_poczatkowa_pocisku;
		Inicjalizacja();
	}

	void Inicjalizacja() {

		UstawieniePoczatkowe(v_pocisku, polozenia[0],predkosci[0],false);
	}

	double Symulacja() {
		
		return KontrolerTrzeciaPredkoscKosmiczna();
	
	}

	double KontrolerTrzeciaPredkoscKosmiczna() {

		std::ofstream plik("pocisk", std::ios::app);
		//plik << "start" << std::endl;
		const int maxLiczbaProbek = 10;
		static int liczbaProbek = 0;
		static double zarejestrowane_predkosci[maxLiczbaProbek];
		static double odleglosc_Neptun_Slonce = 0.0;
		static double poprzednia_odleglosc_pocisk_slonce = 0.0;
		static PunktMaterialny* p = NULL;
		static PunktMaterialny* s = NULL;
		static Wektor pocisk_polozenie = Wektor(0, 0, 0);
		static double krok_predkosci = 0.02;

		if (liczbaProbek < maxLiczbaProbek) {

			if (!flaga_pocisk) {

				UstawPozycjePoczatkowaIPredkosc(v_pocisku, polozenia[liczbaProbek],predkosci[liczbaProbek], true);
				PunktMaterialny* neptun = PobierzPunktMaterialny(ilosc - 2);
				s = PobierzPunktMaterialny(0);
				p = PobierzPunktMaterialny(ilosc - 1);
				pocisk_polozenie = p->Polozenie();
				odleglosc_Neptun_Slonce = Roznica(neptun->Polozenie(), s->Polozenie()).Dlugosc();
				flaga_pocisk = true;
				
			}
			else 
			{
				bool ponownie_rozmiesc_obiekty = false;
				Wektor aktualne_polozenie_pocisku = p->Polozenie();
				Wektor aktualne_polozenie_slonca = s->Polozenie();
				double aktualna_odleglosc_pocisk_slonce = Roznica(aktualne_polozenie_pocisku, aktualne_polozenie_slonca).Dlugosc();
				if (aktualna_odleglosc_pocisk_slonce > odleglosc_Neptun_Slonce) {

					zarejestrowane_predkosci[liczbaProbek] = v_pocisku;
					liczbaProbek++;
					v_pocisku = predkosc_poczatkowa_pocisku;
					ponownie_rozmiesc_obiekty = true;
				}
				else if (aktualna_odleglosc_pocisk_slonce < poprzednia_odleglosc_pocisk_slonce) {

					v_pocisku += krok_predkosci;
					ponownie_rozmiesc_obiekty = true;
				}

				poprzednia_odleglosc_pocisk_slonce = aktualna_odleglosc_pocisk_slonce;

				if (ponownie_rozmiesc_obiekty) {

					poprzednia_odleglosc_pocisk_slonce = 0.0;
					UstawPozycjePoczatkowaIPredkosc(v_pocisku, polozenia[liczbaProbek%10],predkosci[liczbaProbek%10], true);
					return -1;
				}
			}

		}
		else 
		{
			double sumaPredkosci = 0.0;
			plik << "Zarejestrowane predkosci: " << std::endl;
			for (int i = 0; i < liczbaProbek; ++i) {
				plik << zarejestrowane_predkosci[i] <<"( "<<zarejestrowane_predkosci[i]*4.756<<" km/s)" << std::endl;
				sumaPredkosci += zarejestrowane_predkosci[i];
			}
		
			double wynik = sumaPredkosci / liczbaProbek;
			PunktMaterialny* Ziemia = PobierzPunktMaterialny(3);
			double v_Ziemi = Ziemia->Predkosc().Dlugosc();
			plik << "predkosc Ziemi: " << v_Ziemi*4.756<<" km/s" << std::endl;
			wynik += v_Ziemi;
			flaga_pocisk = false;
			return wynik*4.756;
		}

		return 0.0;
	}

	double suma_promieni_planet() {

		double suma = 0;

		for (int i = 0; i < ilosc - 1; ++i) {

			suma += tablicaPlanet[i]->PromiePlanety();
		}

		return suma;
	}
	void UstawieniePoczatkowe(double szybkosc_pocisku,const double polozenie_planet[9][3],const double predkosci_planet[8][3],bool symulacja) {

		PunktMaterialny* Soce = PobierzPunktMaterialny(0);
		Soce->UstawPolozenie(pooenieSoca);
		Soce->UstawPredkosc(Wektor(0, 0, 0));
		Soce->UstawMase(tablicaPlanet[0]->masa);
		Soce->UstawPromien((0.35*tablicaPlanet[0]->PromiePlanety()) / SumaPromieni);

		for (int i = 1; i < ilosc-1; ++i)
		{
			double polozenieX = polozenie_planet[i - 1][0];
			double polozenieY = polozenie_planet[i - 1][1];
			double polozenieZ = polozenie_planet[i - 1][2];
			
			PunktMaterialny* punkt = PobierzPunktMaterialny(i);
			punkt->UstawPolozenie(Wektor(polozenieX, polozenieY, polozenieZ));

			punkt->UstawPredkosc(365.0*Wektor(predkosci_planet[i - 1][0], predkosci_planet[i - 1][1],predkosci_planet[i - 1][2]));
			punkt->UstawMase(tablicaPlanet[i]->masa);
			punkt->UstawPromien(tablicaPlanet[i]->PromiePlanety() / SumaPromieni);	
			punkt->UstawKolor(i / (float)ilosc, 1 - i / (float)ilosc, 1);

		}
		
	
		PunktMaterialny* Ziemia = PobierzPunktMaterialny(3);
		PunktMaterialny* punkt = PobierzPunktMaterialny(ilosc - 1);
		punkt->UstawPolozenie(Ziemia->Polozenie() + Wektor(5.0058756049e-3, 5.05681e-3, 0));

		if(symulacja)punkt->UstawPredkosc(Wektor(Ziemia->Predkosc().X + szybkosc_pocisku, 0, 0));
		else punkt->UstawPredkosc(Wektor(0, 0, 0));
		
		punkt->UstawMase(tablicaPlanet[ilosc - 1]->masa);
		punkt->UstawPromien(tablicaPlanet[9]->PromiePlanety() / SumaPromieni);
		punkt->UstawKolor(ilosc-1 / (float)ilosc, 1 - ilosc-1 / (float)ilosc, 1);

	}
	
	void UstawPozycjePoczatkowaIPredkosc(double szybkosc_pocisku, const double polozenie_planet[9][3],const double predkosci_planet[8][3], bool symulacja) {

		PunktMaterialny* Soce = PobierzPunktMaterialny(0);
		Soce->UstawPolozenie(pooenieSoca);
		Soce->UstawPredkosc(Wektor(0, 0, 0));

		for (int i = 1; i < ilosc - 1; ++i)
		{
			double polozenieX = polozenie_planet[i - 1][0];
			double polozenieY = polozenie_planet[i - 1][1];
			double polozenieZ = polozenie_planet[i - 1][2];

			PunktMaterialny* punkt = PobierzPunktMaterialny(i);
			punkt->UstawPolozenie(Wektor(polozenieX, polozenieY, polozenieZ));

			punkt->UstawPredkosc(365.0*Wektor(predkosci_planet[i - 1][0], predkosci_planet[i - 1][1], predkosci_planet[i - 1][2]));

		}

		PunktMaterialny* Ziemia = PobierzPunktMaterialny(3);
		PunktMaterialny* punkt = PobierzPunktMaterialny(ilosc - 1);
		punkt->UstawPolozenie(Ziemia->Polozenie() + Wektor(5.0058756049e-3, 5.05681e-3, 0));


		if (symulacja)punkt->UstawPredkosc(Wektor(Ziemia->Predkosc().X + szybkosc_pocisku, 0, 0));
		else punkt->UstawPredkosc(Wektor(0, 0, 0));

	}

	Wektor Roznica(Wektor p2, Wektor p1) const
	{
		Wektor roznica = p2 - p1;
		return roznica;
	}

	Wektor Sila(int indeks) const
	{
		PunktMaterialny* punkt = PobierzPunktMaterialny(indeks);
		int ile = ilosc;
		Wektor sila;
		if (!flaga_pocisk && indeks == 9) return Wektor(0, 0, 0);
		if (!flaga_pocisk) ile--;
		for (int i = 0; i < ile; ++i) {
		
			if (i != indeks)
			{
				PunktMaterialny* punktI = PobierzPunktMaterialny(i);
				Wektor r12 = Roznica(punktI->Polozenie(), punkt->Polozenie());
				double odleglosc2 = SQR(r12.Dlugosc());
				r12.Normuj();
				sila += (punkt->Masa()*punktI->Masa() / odleglosc2)*r12;

			}

		}
	
		return G*sila;
	}
};

#endif