﻿using Windows.Storage;
using Windows.UI.Popups;
using Windows.UI.Xaml;
using Windows.UI.Xaml.Controls;
using Windows.UI.Xaml.Controls.Primitives;
using LightsOut;
using LightsOut.Data;
using LightsOut.Events;
using LightsOut.Interfaces;
using MiniLightsOut.Helpers;

namespace MiniLightsOut.Controls
{
    public partial class GameBoard : UserControl
    {
        #region Public properties
        public string PlayerName { get; set; }
        #endregion

        #region Private fields
        private readonly ILightsOutGame _lightsOutGame;
        private readonly IScoreManager _scoreManager;
        #endregion

        #region Constructors
        public GameBoard()
        {
            this.InitializeComponent();
            this.IsEnabled = false;

            _lightsOutGame = new LightsOutGame();
            _scoreManager = new ScoreManager();
            _scoreManager.ReadFromFileAsync();

            InitializeGame();
            _lightsOutGame.GameEnded += LightsOutGame_OnGameEnded;
            _lightsOutGame.LightSwitched += LightsOutGame_OnLightSwitched;
            ApplicationData.Current.DataChanged += ApplicationData_OnDataChanged;
        }
        #endregion

        #region Public methods
        public void NewGame()
        {
            _lightsOutGame.NewGame();
            this.IsEnabled = true;
        }
        #endregion

        #region Private methods
        private void SetColumnDefinitions()
        {
            GridMain.ColumnDefinitions.Clear();
            for (int i = 0; i < _lightsOutGame.ColumnCount; i++)
            {
                var columnDefinition = new ColumnDefinition
                {
                    Width = new GridLength(1, GridUnitType.Star)
                };
                GridMain.ColumnDefinitions.Add(columnDefinition);
            }
        }

        private void SetRowDefinitions()
        {
            GridMain.RowDefinitions.Clear();
            for (int i = 0; i < _lightsOutGame.RowCount; i++)
            {
                var rowDefinition = new RowDefinition { Height = new GridLength(1, GridUnitType.Star) };
                GridMain.RowDefinitions.Add(rowDefinition);
            }
        }

        /// <summary>
        /// Initialize the grid then create buttons representing lights and add them to the grid.
        /// </summary>
        private void InitializeGame()
        {
            // Clear the grid and set row/column definitions
            GridMain.Children.Clear();
            SetRowDefinitions();
            SetColumnDefinitions();

            // Add toggle buttons representing lights to the grid
            for (int i = 0; i < _lightsOutGame.RowCount; i++)
            {
                for (int j = 0; j < _lightsOutGame.ColumnCount; j++)
                {
                    var newButton = new ToggleButton();
                    Grid.SetRow(newButton, i);
                    Grid.SetColumn(newButton, j);
                    newButton.HorizontalAlignment = HorizontalAlignment.Stretch;
                    newButton.VerticalAlignment = VerticalAlignment.Stretch;
                    newButton.Click += ToggleButton_OnClick;
                    newButton.Style = GridMain.Resources["ButtonLightStyle"] as Style;
                    GridMain.Children.Add(newButton);
                }
            }
        }
        #endregion

        #region Events
        protected async virtual void ApplicationData_OnDataChanged(ApplicationData sender, object args)
        {
            var message = "Inne urządzenie nadpisało plik z najlepszymi wynikami! Czy chcesz wczytać ten plik i nadpisać obecne wyniki?";
            var clickedYes = await MessageDialogHelper.DisplayYesNoDialogAsync(message);
            if (clickedYes)
            {
                await _scoreManager.ReadFromFileAsync();
            }
            else
            {
                await _scoreManager.WriteToFileAsync();
            }
        }

        protected virtual void LightsOutGame_OnLightSwitched(object sender, LightSwitchedEventArgs e)
        {
            var row = e.Row;
            var column = e.Column;

            var gridIndex = row * _lightsOutGame.ColumnCount + column;
            var button = (ToggleButton)GridMain.Children[gridIndex];
            button.IsChecked = _lightsOutGame[row, column];
        }

        protected virtual void LightsOutGame_OnGameEnded(object sender, GameEndedEventArgs e)
        {
            this.IsEnabled = false;

            var score = new PlayerScore(PlayerName, _lightsOutGame.MoveCount, e.TimeSpan.Ticks);
            _scoreManager.AddScoreAsync(score);

            var formattedString = "Gra zakonczona!\n";
            formattedString += string.Format("Ilość ruchów: {0}\n", _lightsOutGame.MoveCount);
            formattedString += string.Format("Czas: {0:D2}:{1:D2}:{2:D2}",
                e.TimeSpan.Minutes, e.TimeSpan.Seconds, e.TimeSpan.Milliseconds);

            var dialog = new MessageDialog(formattedString);
            dialog.ShowAsync();
        }

        private void ToggleButton_OnClick(object sender, RoutedEventArgs e)
        {
            var toggleButton = (ToggleButton)sender;
            var row = Grid.GetRow(toggleButton);
            var column = Grid.GetColumn(toggleButton);
            _lightsOutGame.PlayerAction(row, column);
        }
        #endregion
    }
}
