﻿using System;
using System.ComponentModel;
using Windows.UI.Xaml.Controls;
using Windows.Devices.Geolocation;
using Windows.UI;
using Windows.UI.Xaml;
using Windows.UI.Xaml.Media;
using Windows.UI.Xaml.Shapes;
using System.Threading;
using System.Threading.Tasks;

#if WINDOWS_APP
using Bing.Maps;
using Bing.Maps.Search;
using System.Collections.Generic;
#elif WINDOWS_PHONE_APP
using Windows.UI.Xaml.Controls.Maps;
using Windows.Services.Maps;
using System.Collections.Generic;
#endif

namespace DeliveryHelper
{
    public class UniversalMap : Grid, INotifyPropertyChanged
    {
#if WINDOWS_APP
        private Bing.Maps.Map _map;
#elif WINDOWS_PHONE_APP
        private Windows.UI.Xaml.Controls.Maps.MapControl _map;
#endif

        public UniversalMap()
        {
#if WINDOWS_APP
            _map = new Bing.Maps.Map() { };
#elif WINDOWS_PHONE_APP
            _map = new MapControl();
#endif

            this.Children.Add(_map);
        }

        public void Register(string token)
        {
            #if WINDOWS_APP
            _map.Credentials = token;
#elif WINDOWS_PHONE_APP
            _map.MapServiceToken = token;
#endif
        }

        public void SetView(Geopoint center, double zoom = 7)
        {
#if WINDOWS_APP
            _map.SetView(center.Position.ToLocation(), zoom);
#elif WINDOWS_PHONE_APP
            _map.Center = center;
            _map.ZoomLevel = zoom;
#endif
        }

        public async void SetViewAll(List<Geopoint> List)
        {
#if WINDOWS_APP
            LocationCollection Locs = new LocationCollection();

            foreach (var Point in List)
            {
                if (Point != null)
                {
                    Locs.Add(Point.Position.ToLocation());
                }
            }

            var rect = new LocationRect(Locs);
            rect.Width *= 1.3;
            rect.Height *= 1.3;

            float MinSize = 0.0025f;

            if (rect.Height < MinSize)
            {
                rect.Height = MinSize;
            }

            if (rect.Width < MinSize)
            {
                rect.Width = MinSize;
            }

            _map.SetView(rect);
#elif WINDOWS_PHONE_APP
            List<BasicGeoposition> Locs = new List<BasicGeoposition>();

            foreach (var Point in List)
            {
                if (Point != null)
                {
                    Locs.Add(Point.Position);
                }
            }

            GeoboundingBox Box = GeoboundingBox.TryCompute(Locs);
            await _map.TrySetViewBoundsAsync(Box, new Thickness(5), MapAnimationKind.Bow);
#endif
        }



        public Geopoint GetCenter()
        {
#if WINDOWS_APP
            return _map.Center.ToGeopoint();
#elif WINDOWS_PHONE_APP
            return _map.Center;
#endif
        }

        public enum PinColor { Blue, Red, Green, Yellow, Black }

        public Grid AddPushpin(Geopoint point, string text, PinColor color = PinColor.Blue)
        {
            int PinSize = 24;

            Brush FillColor;

            switch (color)
            {
                case PinColor.Blue:
                default:
                    FillColor = new SolidColorBrush(Colors.DodgerBlue);
                    break;
                case PinColor.Red:
                    FillColor = new SolidColorBrush(Colors.Red);
                    break;
                case PinColor.Green:
                    FillColor = new SolidColorBrush(Colors.Green);
                    break;
                case PinColor.Yellow:
                    FillColor = new SolidColorBrush(Colors.Yellow);
                    break;
                case PinColor.Black:
                    FillColor = new SolidColorBrush(Colors.Black);
                    break;
            }
            var Pin = new Grid()
            {
                Width = PinSize,
                Height = PinSize,
                Margin = new Windows.UI.Xaml.Thickness(-PinSize * 0.5)
            };

            Pin.Children.Add(new Ellipse()
            {
                Fill = FillColor,
                Stroke = new SolidColorBrush(Colors.White),
                StrokeThickness = 3,
                Width = PinSize,
                Height = PinSize
            });

            Pin.Children.Add(new TextBlock()
            {
                Text = text,
                //FontSize = 12,
                //Foreground = new SolidColorBrush(Colors.White),
                HorizontalAlignment = Windows.UI.Xaml.HorizontalAlignment.Center,
                VerticalAlignment = Windows.UI.Xaml.VerticalAlignment.Center
            });

#if WINDOWS_APP
            MapLayer.SetPosition(Pin, point.Position.ToLocation());
#elif WINDOWS_PHONE_APP
            MapControl.SetLocation(Pin, point);
#endif
            _map.Children.Add(Pin);
            return Pin;
        }

        public void RemovePin(Grid Pin)
        {
            _map.Children.Remove(Pin);
        }



        public async Task<Geopoint> GetByAddress(string Address)
        {
#if WINDOWS_APP
            // dodac przestrzen nazw using Bing.Maps.Search; using Windows.Foundation; using System.Threading; using System.Threading.Tasks;
            GeocodeRequestOptions GeocodeRequest = new GeocodeRequestOptions(Address);
            LocationDataResponse Response = await _map.SearchManager.GeocodeAsync(GeocodeRequest);

            if (!Response.HasError && Response.LocationData.Count > 0)
            {
                var res = Response.LocationData[0].Location.ToGeopoint();
                return res;
            }
#elif WINDOWS_PHONE_APP
            //using Windows.Services.Maps;

            Geopoint Hint = _map.Center;
            MapLocationFinderResult result = await MapLocationFinder.FindLocationsAsync(Address, Hint, 1);

            if (result.Status == MapLocationFinderStatus.Success && result.Locations.Count > 0)
            {
                return result.Locations[0].Point;
            }
#endif
            return null;
        }



        public async Task AddRouteSimple(List<Geopoint> Points)
        {
#if WINDOWS_APP
            _map.DirectionsManager.ClearActiveRoute();
            _map.DirectionsManager.Waypoints.Clear();

            _map.DirectionsManager.RequestOptions.RouteMode = Bing.Maps.Directions.RouteModeOption.Driving;
            _map.DirectionsManager.RequestOptions.DistanceUnit = Bing.Maps.Directions.DistanceUnitOption.Kilometer;
            _map.DirectionsManager.RequestOptions.Optimize = Bing.Maps.Directions.OptimizeOption.Time;
            _map.DirectionsManager.RequestOptions.Avoid = Bing.Maps.Directions.AvoidOption.Highways | Bing.Maps.Directions.AvoidOption.ExpressTrain;

            foreach (Geopoint Gpt in Points)
            {
                _map.DirectionsManager.Waypoints.Add(new Bing.Maps.Directions.Waypoint(Gpt.Position.ToLocation()) /*{ IsViaPoint = true }*/);
            }

            _map.DirectionsManager.RenderOptions.ActiveRoutePolylineOptions.LineColor = Colors.Peru;
            _map.DirectionsManager.RenderOptions.ActiveRoutePolylineOptions.LineWidth = 10.0;
            _map.DirectionsManager.RenderOptions.WaypointPushpinOptions.Visible = true;
            _map.DirectionsManager.RenderOptions.StepPushpinOptions.Visible = true;

            try
            {
                await _map.DirectionsManager.CalculateDirectionsAsync();
                var LengthType = _map.DirectionsManager.ActiveRoute.DistanceUnit;
                var Length = _map.DirectionsManager.ActiveRoute.TravelDistance;
            }
            catch (Exception e)
            {
                e.ToString();
            }

#elif WINDOWS_PHONE_APP
            _map.Routes.Clear();

            MapRouteFinderResult RouteResult = await MapRouteFinder.GetDrivingRouteFromWaypointsAsync(Points, MapRouteOptimization.Time, MapRouteRestrictions.DirtRoads);

            if (RouteResult.Status == MapRouteFinderStatus.Success)
            {
                MapRouteView RouteView = new MapRouteView(RouteResult.Route);

                RouteView.RouteColor = Colors.Blue;

                RouteView.OutlineColor = Colors.Red;
                //RouteView.Route.
                _map.Routes.Add(RouteView);
            }
#endif
        }

        public void ClearRoutesTest()
        {
        }



        public async Task AddRoute(List<Geopoint> Points, bool setView = false)
        {
            if (Points.Count < 2)
                return;

#if WINDOWS_APP
            //_map.DirectionsManager.ClearActiveRoute();
            _map.DirectionsManager.Waypoints.Clear();
            _map.DirectionsManager.HideRoutePath(DeliveryMapHelper.Instance.FirstSegment);
            _map.DirectionsManager.HideRoutePath(DeliveryMapHelper.Instance.RestSegment);
            //_map.DirectionsManager.ActiveRoute = DeliveryMapHelper.Instance.FirstSegment;
            //_map.DirectionsManager.ClearActiveRoute();
            //_map.DirectionsManager.ActiveRoute = DeliveryMapHelper.Instance.RestSegment;
            //_map.DirectionsManager.ClearActiveRoute();
            DeliveryMapHelper.Instance.FirstSegment = null;
            DeliveryMapHelper.Instance.RestSegment = null;


            _map.DirectionsManager.RequestOptions.RouteMode = Bing.Maps.Directions.RouteModeOption.Driving;
            _map.DirectionsManager.RequestOptions.DistanceUnit = Bing.Maps.Directions.DistanceUnitOption.Kilometer;
            _map.DirectionsManager.RequestOptions.Optimize = Bing.Maps.Directions.OptimizeOption.Time;

            _map.DirectionsManager.RequestOptions.FormattedInstruction = true;
            
            _map.DirectionsManager.RenderOptions.AutoUpdateMapView = setView;
            _map.DirectionsManager.RenderOptions.WaypointPushpinOptions.Visible = true;
            _map.DirectionsManager.RenderOptions.StartWaypointColorBrush = new SolidColorBrush(Colors.Purple);
            //var DefaultColor = _map.DirectionsManager.RenderOptions.ActiveRoutePolylineOptions.LineColor;

            _map.DirectionsManager.Waypoints.Add(new Bing.Maps.Directions.Waypoint(Points[0].Position.ToLocation()));
            _map.DirectionsManager.Waypoints.Add(new Bing.Maps.Directions.Waypoint(Points[1].Position.ToLocation()));
            //_map.DirectionsManager.RenderOptions.ActiveRoutePolylineOptions.LineColor = Colors.Red;

            _map.DirectionsManager.RenderOptions.AutoSetActiveRoute = false;

            try
            {
                await _map.DirectionsManager.CalculateDirectionsAsync();
            }
            catch (Exception e)
            {
                e.ToString();
            }
            Bing.Maps.Directions.Route FirstSegment;
            DeliveryMapHelper.Instance.FirstSegment = _map.DirectionsManager.RouteResult[0];
            FirstSegment = DeliveryMapHelper.Instance.FirstSegment; 

            Points.Remove(Points[0]);
            
            _map.DirectionsManager.Waypoints.Clear();
            //_map.DirectionsManager.RenderOptions.ActiveRoutePolylineOptions.LineColor = DefaultColor;
                        
            foreach (Geopoint Gpt in Points)
            {
                _map.DirectionsManager.Waypoints.Add(new Bing.Maps.Directions.Waypoint(Gpt.Position.ToLocation()) /*{ IsViaPoint = true }*/);
            }

            try
            {
                await _map.DirectionsManager.CalculateDirectionsAsync();
            }
            catch (Exception e)
            {
                e.ToString();
            }

            _map.DirectionsManager.ClearActiveRoute();

            Bing.Maps.Directions.Route RestSegment;

            DeliveryMapHelper.Instance.RestSegment = _map.DirectionsManager.RouteResult[0];
            RestSegment = DeliveryMapHelper.Instance.RestSegment; 


            FirstSegment.RoutePath.LineColor = Colors.Red;
            FirstSegment.RoutePath.LineWidth *= 1.2;

            _map.DirectionsManager.ShowRoutePath(RestSegment);
            _map.DirectionsManager.ShowRoutePath(FirstSegment);

            //_map.DirectionsManager.ActiveRoute = FirstSegment;
            //_map.DirectionsManager.ClearActiveRoute();

            //_map.DirectionsManager.HideRoutePath(FirstSegment);
#elif WINDOWS_PHONE_APP
            _map.Routes.Clear();

            var RouteresultFirst = await MapRouteFinder.GetDrivingRouteAsync(Points[0], Points[1], MapRouteOptimization.Time, MapRouteRestrictions.DirtRoads);
            if (RouteresultFirst.Status == MapRouteFinderStatus.Success)
            {
                var ModRouteFirst = RouteresultFirst.Route;
                MapRouteView RouteViewFirst = new MapRouteView(RouteresultFirst.Route);
                RouteViewFirst.OutlineColor = Colors.Red;

                Points.Remove(Points[0]);

                var RouteResult = await MapRouteFinder.GetDrivingRouteFromWaypointsAsync(Points, MapRouteOptimization.Time, MapRouteRestrictions.DirtRoads);

                if (RouteResult.Status == MapRouteFinderStatus.Success)
                {
                    var ModRoute = RouteResult.Route;
                    MapRouteView RouteView = new MapRouteView(RouteResult.Route);

                    RouteView.RouteColor = Colors.Blue;

                    _map.Routes.Add(RouteView);
                    _map.Routes.Add(RouteViewFirst);
                }

            }
#endif
        }

        public event PropertyChangedEventHandler PropertyChanged;

        internal void OnPropertyChanged(string propertyName)
        {
            if (PropertyChanged != null)
            {
                PropertyChanged(this, new System.ComponentModel.PropertyChangedEventArgs(propertyName));
            }
        }
    }
}