﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.Globalization;
using System.IO;
using System.Linq;
using System.Linq.Expressions;
using System.Reflection;
using System.Text;
using System.Threading.Tasks;

namespace LinqToCsv
{
    public class CsvException : Exception
    {
        public CsvException(string message, Exception innerException = null)
            :base(message, innerException)
        {
        }
    }

    public interface ICsvRecord
    {
        void ParseValues(string[] values, IFormatProvider formatProvider);
        string[] ToValues(IFormatProvider formatProvider);
    }

    public class CsvDocument<T> : IEnumerable<T>
        where T : ICsvRecord, new()
    {
        private static readonly IFormatProvider formarProvider = CultureInfo.InvariantCulture;

        protected List<T> records = new List<T>();

        protected string ścieżkaPliku = null;
        protected char separatorChar = ',';

        public int Count
        {
            get
            {
                return records.Count;
            }
        }

        public T[] GetRecords()
        {
            return records.ToArray();
        }

        public CsvDocument(IEnumerable<T> records = null)
        {
            //this.records.Clear();
            if (records != null) this.records.AddRange(records);
        }

        private static bool TryParseRecordSecured(ref T record, string line, char separatorChar)
        {
            try
            {
                string[] strings = line.Split(separatorChar);
                record.ParseValues(strings, formarProvider);
                return true;
            }
            catch (Exception exc)
            {
                return false;
            }
        }

        private static string RecordToLineSecures(T record, char separatorChar)
        {
            string line = "";
            foreach(string value in record.ToValues(formarProvider))
            {
                line += value + separatorChar;
            }
            line.TrimEnd(separatorChar);
            return line;
        }

        public static CsvDocument<T> Load(string ścieżkaPliku, char separatorChar = ',', Func<T, bool> filtr = null)
        {
            CsvDocument<T> csv = new CsvDocument<T>();
            using (StreamReader sr = new StreamReader(ścieżkaPliku))
            {
                string line;
                while ((line = sr.ReadLine()) != null)
                {
                    T record = new T();
                    if (TryParseRecordSecured(ref record, line, separatorChar))
                    {
                        if (filtr == null || filtr(record)) csv.records.Add(record);
                    }
                    else throw new CsvException("Błąd podczas parsowania linii " + line);
                }
            }
            csv.ścieżkaPliku = ścieżkaPliku;
            csv.separatorChar = separatorChar;
            return csv;
        }

        public void AddRecord(T record)
        {
            records.Add(record);
        }

        public void RemoveRecord(T record)
        {
            records.Remove(record);
        }

        public T this[int index]
        {
            get
            {
                return records[index];
            }
            set
            {
                records[index] = value;
            }
        }

        public void UpdateRecord(int index, T record)
        {
            this[index] = record;
        }

        public void SaveAs(string ścieżkaPliku, char separatorChar = ',')
        {
            using (StreamWriter sw = new StreamWriter(ścieżkaPliku, false, Encoding.UTF8))
            {
                foreach(T record in records)
                {
                    sw.WriteLine(RecordToLineSecures(record, separatorChar));
                }
            }

                this.ścieżkaPliku = ścieżkaPliku;
            this.separatorChar = separatorChar;
        }

        public void Save()
        {
            SaveAs(ścieżkaPliku, separatorChar);
        }

        public virtual IEnumerator<T> GetEnumerator()
        {
            return records.GetEnumerator();
        }

        IEnumerator IEnumerable.GetEnumerator()
        {
            return this.GetEnumerator();
        }
    }

    public class CsvDocumentQueryable<T> : CsvDocument<T>, IOrderedQueryable<T>
        where T : ICsvRecord, new()
    {
        public CsvDocumentQueryable(string ścieżkaPliku, char separatorChar = ',')
            //: base()
        {
            this.ścieżkaPliku = ścieżkaPliku;
            this.separatorChar = separatorChar;

            expression = Expression.Constant(this);
            provider = new CsvQueryProvider<T>(ścieżkaPliku, separatorChar);
        }

        internal CsvDocumentQueryable(IQueryProvider provider, Expression expression)
        {
            if (provider == null) throw new ArgumentNullException(nameof(provider));
            else this.provider = provider;

            if (expression == null) throw new ArgumentNullException(nameof(expression));
            else this.expression = expression;
        }

        private IQueryProvider provider { get; set; }
        private Expression expression { get; set; }

        Type IQueryable.ElementType
        {
            get
            {
                return typeof(T);
            }
        }

        public Expression Expression
        {
            get
            {
                return expression;
            }
        }

        public IQueryProvider Provider
        {
            get
            {
                return provider;
            }
        }

        public override IEnumerator<T> GetEnumerator()
        {
            return (provider.Execute<IEnumerable<T>>(Expression)).GetEnumerator();
        }

        IEnumerator IEnumerable.GetEnumerator()
        {
            return (provider.Execute<IEnumerable>(Expression)).GetEnumerator();
        }
    }


    public class CsvAutoRecord<T> : ICsvRecord
        where T : new()
    {
        private T record = default(T);

        public T Value
        {
            get
            {
                return record;
            }
        }

        private static bool isProper(FieldInfo fieldInfo)
        {
            return true;
        }

        public static FieldInfo[] getFields()
        {
            FieldInfo[] fields = typeof(T).GetFields().Where(isProper).ToArray();
            return fields;
        }

        private static FieldInfo[] fields = getFields();

        public CsvAutoRecord(T record) 
        {
            this.record = record;
        }

        public CsvAutoRecord() { } //new() - wymagany przez CsvDocument

        public void ParseValues(string[] values, IFormatProvider formatProvider)
        {
            List<string> list = values.ToList();
            record = new T();
            foreach (FieldInfo fi in fields)
            {
                string s = list.First();
                object o = Convert.ChangeType(s, fi.FieldType, formatProvider);
                fi.SetValue(record, o);
                list.RemoveAt(0);
            }
        }

        public string[] ToValues(IFormatProvider formatProvider)
        {
            List<string> values = new List<string>();
            foreach (FieldInfo fi in fields)
            {
                object o = fi.GetValue(record);
                string s = Convert.ToString(o, formatProvider);
                values.Add(s);
            }
            return values.ToArray();
        }
    }
}
