﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Linq.Expressions;
using System.Text;
using System.Threading.Tasks;

namespace LinqToCsv
{
    public class CsvQueryProvider<T> : IQueryProvider
        where T : ICsvRecord, new()
    {
        string ścieżkaPliku;
        char separatorChar;

        public CsvQueryProvider(string ścieżkaPliku, char separatorChar = ',')
        {
            this.ścieżkaPliku = ścieżkaPliku;
            this.separatorChar = separatorChar;
        }

        public IQueryable CreateQuery(Expression expression)
        {
            return new CsvDocumentQueryable<T>(this, expression);
        }

        public IQueryable<TElement> CreateQuery<TElement>(Expression expression)
        {
            CsvDocumentQueryable<T> csv = new CsvDocumentQueryable<T>(this, expression);
            if (typeof(TElement) == typeof(T)) return (IQueryable<TElement>)csv;
            //if (typeof(TElement) == typeof(string)) return (IQueryable<TElement>)csv.Select((T element) => { return element.ToString(); }).AsQueryable();
            else throw new Exception("Nieodpowiedni typ zwracany przez zapytanie");
        }

        private object _Execute(Expression expression, bool isEnumerable)
        {
            if (!isQueryOverDataSource(expression)) throw new InvalidProgramException("No query over data source was specified");

            InnermostClauseFinder whereFinder = new InnermostClauseFinder("Where");
            MethodCallExpression whereExpression = whereFinder.GetInnermostMethodCall(expression);
            LambdaExpression whereLambda = (LambdaExpression)((UnaryExpression)(whereExpression.Arguments[1])).Operand;
            Func<T, bool> filtr = (Func < T, bool>)whereLambda.Compile();

            T[] records = CsvDocument<T>.Load(ścieżkaPliku, separatorChar, filtr).GetRecords();

            InnermostClauseFinder orderByFinder = new InnermostClauseFinder("OrderBy");
            MethodCallExpression orderbyExpression = orderByFinder.GetInnermostMethodCall(expression);
            LambdaExpression orderbyLambda = (LambdaExpression)((UnaryExpression)(orderbyExpression.Arguments[1])).Operand;
            Delegate selectKey = orderbyLambda.Compile();

            object[] keys = new object[records.Length];
            for (int i = 0; i < keys.Length; ++i) keys[i] = selectKey.DynamicInvoke(records[i]);
            Array.Sort(keys, records);

            IQueryable<T> queryableRecords = records.AsQueryable();

            ExpressionTreeModifier<T> treeCopier = new ExpressionTreeModifier<T>(queryableRecords);
            Expression newExpressionTree = treeCopier.Visit(expression);

            if (isEnumerable) return queryableRecords.Provider.CreateQuery(newExpressionTree);
            else return queryableRecords.Provider.Execute(newExpressionTree);
        }

        private static bool isQueryOverDataSource(Expression expression)
        {
            return expression is MethodCallExpression;
        }

        public object Execute(Expression expression)
        {
            return _Execute(expression, false);
        }

        public TResult Execute<TResult>(Expression expression)
        {
            bool isEnumerable = (typeof(TResult).Name == "IEnumerable`1");
            return (TResult)_Execute(expression, isEnumerable);
        }
    }

    internal class ExpressionTreeModifier<T> : ExpressionVisitor
        where T : ICsvRecord, new()
    {
        private IQueryable<T> queryableRecords;

        internal ExpressionTreeModifier(IQueryable<T> records)
        {
            this.queryableRecords = records;
        }

        protected override Expression VisitConstant(ConstantExpression c)
        {
            // Replace the constant QueryableTerraServerData arg with the queryable Place collection. 
            if (c.Type == typeof(CsvDocumentQueryable<T>))
                return Expression.Constant(this.queryableRecords);
            else
                return c;
        }
    }

    class InnermostClauseFinder : ExpressionVisitor
    {
        string clauseName;

        public InnermostClauseFinder(string clauseName)
        {
            this.clauseName = clauseName;
        }

        private MethodCallExpression innermostMethodCall;

        public MethodCallExpression GetInnermostMethodCall(Expression expression)
        {
            Visit(expression);
            return innermostMethodCall;
        }

        protected override Expression VisitMethodCall(MethodCallExpression expression)
        {
            if (expression.Method.Name == clauseName)
                innermostMethodCall = expression;

            Visit(expression.Arguments[0]);

            return expression;
        }
    }
}
