﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Drawing;
using System.Drawing.Imaging;
using System.Windows.Forms;

namespace Przegladarka {
   
   class BitmapEditor {
      
      private int bytesPerPixel;
      private int heightInPixels;
      private int widthInBytes;

      private int CheckPixelFormat(Bitmap bitmap) {

         int pixelFormat = Bitmap.GetPixelFormatSize(bitmap.PixelFormat);

         switch (pixelFormat) {
            case 32:
            case 24:
               return pixelFormat;
            default:
               MessageBox.Show(string.Format("Unsupported Format. Pixel format: {0}. Pixel format must be 24 or 32 bits.", pixelFormat.ToString()));
               return 0;
         }
      }

      private bool ValidatePixelFormat(Bitmap bitmap) {
         int pixelFormat = CheckPixelFormat(bitmap);
         if (pixelFormat == 0)
            return false;
         else
            return true;
      }

      private void GetBitmapData(Bitmap bitmap) {
         BitmapData bitmapData = bitmap.LockBits(new Rectangle(0, 0, bitmap.Width, bitmap.Height), ImageLockMode.ReadWrite, bitmap.PixelFormat);
         bytesPerPixel = System.Drawing.Bitmap.GetPixelFormatSize(bitmap.PixelFormat) / 8;
         heightInPixels = bitmapData.Height;
         widthInBytes = bitmapData.Width * bytesPerPixel;
         bitmap.UnlockBits(bitmapData);
      }

      public void NegateColor(Bitmap bitmap, bool useSafe) {
         if (useSafe)
         NegateColorSafe(bitmap);
         else
         NegateColorUnsafe(bitmap);
      }

      private void NegateColorUnsafe(Bitmap bitmap) {

         if (ValidatePixelFormat(bitmap) == false)
            return;

         unsafe {

            GetBitmapData(bitmap);
            
            BitmapData bitmapData = bitmap.LockBits(new Rectangle(0, 0, bitmap.Width, bitmap.Height), ImageLockMode.ReadWrite, bitmap.PixelFormat);
            byte* ptrFirstPixel = (byte*)bitmapData.Scan0;

            for (int h = 0; h < heightInPixels; h++) {
               byte* currentLine = ptrFirstPixel + (h * bitmapData.Stride);

               for (int w = 0; w < widthInBytes; w = w + bytesPerPixel) {
                  int readBlue = currentLine[w];
                  int readGreen = currentLine[w + 1];
                  int readRed = currentLine[w + 2];

                  //int avg = (oldBlue + oldGreen +oldRed)/3;

                  //currentLine[x+2] = (byte)(255);
                  //currentLine[x] = (byte)(avg);
                  //currentLine[x + 1] = (byte)(avg);
                  //currentLine[x + 2] = (byte)(avg);

                  currentLine[w] = (byte)(255 - readBlue);
                  currentLine[w + 1] = (byte)(255 - readGreen);
                  currentLine[w + 2] = (byte)(255 - readRed);
               }
            }
            bitmap.UnlockBits(bitmapData);
         }
      }

      private void NegateColorSafe(Bitmap bitmap) {
         int w, h;
         for (w = 0; w < bitmap.Width; w++) {
            for (h = 0; h < bitmap.Height; h++) {
               Color pixelColor = bitmap.GetPixel(w, h);
               Color newColor = Color.FromArgb(255 - pixelColor.R, 255 - pixelColor.G, 255 - pixelColor.B);
               bitmap.SetPixel(w, h, newColor);
            }
         }

      }

      public void Sharpen(Bitmap bitmap, bool useSafe) {
         if (useSafe)
            SharpenSafe(bitmap);
         else
            SharpenUnsafe(bitmap);
      }

      private void SharpenSafe(Bitmap bitmap ) {

         double sharpFactor = 0.5d;

         int w, h;

         for (w = 1; w < (bitmap.Width - 1); w++) {
            for (h = 1; h < (bitmap.Height - 1); h++) {

               int red = (int)((bitmap.GetPixel(w, h).R + sharpFactor * (bitmap.GetPixel(w, h).R - bitmap.GetPixel(w - 1, h - 1).R)));
               int green = (int)((bitmap.GetPixel(w, h).G + sharpFactor * (bitmap.GetPixel(w, h).G - bitmap.GetPixel(w - 1, h - 1).B)));
               int blue = (int)((bitmap.GetPixel(w, h).B + sharpFactor * (bitmap.GetPixel(w, h).G - bitmap.GetPixel(w - 1, h - 1).B)));

               red = Math.Min(Math.Max(red, 0), 255);
               green = Math.Min(Math.Max(green, 0), 255);
               blue = Math.Min(Math.Max(blue, 0), 255);

               bitmap.SetPixel(w, h, Color.FromArgb(red, green, blue));
            }
         }
      }
      
      private void SharpenUnsafe(Bitmap bitmap) {

         if (ValidatePixelFormat(bitmap) == false)
            return;

         double sharpFactor = 0.5d;
         int red, green, blue;

         unsafe {

            GetBitmapData(bitmap);

            BitmapData bitmapData = bitmap.LockBits(new Rectangle(0, 0, bitmap.Width, bitmap.Height), ImageLockMode.ReadWrite, bitmap.PixelFormat);

            byte* ptrFirstPixel = (byte*)bitmapData.Scan0;

            for (int h = 1; h < heightInPixels - 1; h++) {
               byte* previousLine = ptrFirstPixel + ((h - 1) * bitmapData.Stride);
               byte* currentLine = ptrFirstPixel + (h * bitmapData.Stride);

               for (int w = (0 + 1 * bytesPerPixel); w < (widthInBytes - 1 * bytesPerPixel); w = w + bytesPerPixel) {

                  blue = (int)(currentLine[w] + sharpFactor * (currentLine[w] - previousLine[w - bytesPerPixel]));
                  green = (int)(currentLine[w + 1] + sharpFactor * (currentLine[w + 1] - previousLine[w + 1 - bytesPerPixel]));
                  red = (int)(currentLine[w + 2] + sharpFactor * (currentLine[w + 2] - previousLine[w + 2 - bytesPerPixel]));

                  red = Math.Min(Math.Max(red, 0), 255);
                  green = Math.Min(Math.Max(green, 0), 255);
                  blue = Math.Min(Math.Max(blue, 0), 255);

                  currentLine[w] = (byte)(blue);
                  currentLine[w + 1] = (byte)(green);
                  currentLine[w + 2] = (byte)(red);
               }
            }
            bitmap.UnlockBits(bitmapData);
         }
      }

      public void Smooth(Bitmap bitmap, bool useSafe) {
         if (useSafe)
            SmoothSafe(bitmap);
         else
            SmoothUnsafe(bitmap);
      }

      private void SmoothSafe(Bitmap bitmap) {

         int smoothFactor = 2;
         int smoothAreaSize = (1 + smoothFactor * 2) * (1 + smoothFactor * 2);
         int red, green, blue;

         int w, h;

         for (w = smoothFactor; w < (bitmap.Width - smoothFactor); w++) {
            for (h = smoothFactor; h < (bitmap.Height - smoothFactor); h++) {
               red = 0;
               green = 0;
               blue = 0;

               for (int i = -smoothFactor; i <= smoothFactor; i++) {
                  for (int j = -smoothFactor; j <= smoothFactor; j++) {
                     red += bitmap.GetPixel(w + i, h + i).R;
                     green += bitmap.GetPixel(w + i, h + i).G;
                     blue += bitmap.GetPixel(w + i, h + i).B;
                  }
               }

               red = (int)(red / smoothAreaSize);
               green = (int)(green / smoothAreaSize);
               blue = (int)(blue / smoothAreaSize);

               red = Math.Min(Math.Max(red, 0), 255);
               green = Math.Min(Math.Max(green, 0), 255);
               blue = Math.Min(Math.Max(blue, 0), 255);

               bitmap.SetPixel(w, h, Color.FromArgb(red, green, blue));
            }

            }
         }
   
      private void SmoothUnsafe(Bitmap bitmap) {

         if (ValidatePixelFormat(bitmap) == false)
            return;

         int smoothFactor = 1;
         int dimension = 1 + smoothFactor * 2;
         int smoothAreaSize = (dimension) * (dimension);
         int red, green, blue;

         unsafe {

            GetBitmapData(bitmap);
            
            BitmapData bitmapData = bitmap.LockBits(new Rectangle(0, 0, bitmap.Width, bitmap.Height), ImageLockMode.ReadWrite, bitmap.PixelFormat);

            byte* ptrFirstPixel = (byte*)bitmapData.Scan0;

            byte*[] lines = new byte*[dimension];

            for (int h = smoothFactor; h < (heightInPixels - smoothFactor); h++) {
               for (int d = -smoothFactor; d <= smoothFactor; d++) {
                  lines[d + smoothFactor] = ptrFirstPixel + ((h + d) * bitmapData.Stride); ;
               }

               for (int w = (0 + smoothFactor * bytesPerPixel); w < (widthInBytes - smoothFactor * bytesPerPixel); w = w + bytesPerPixel) {
                  red = 0;
                  green = 0;
                  blue = 0;

                  for (int d = -smoothFactor; d <= smoothFactor; d++) {
                     for (int i = -smoothFactor; i <= smoothFactor; i++) {
                        red += lines[d + smoothFactor][w + 2 + i * bytesPerPixel];
                        green += lines[d + smoothFactor][w + 1 + i * bytesPerPixel];
                        blue += lines[d + smoothFactor][w + i * bytesPerPixel];
                     }
                  }

                  red = (int)(red / smoothAreaSize);
                  green = (int)(green / smoothAreaSize);
                  blue = (int)(blue / smoothAreaSize);

                  red = Math.Min(Math.Max(red, 0), 255);
                  green = Math.Min(Math.Max(green, 0), 255);
                  blue = Math.Min(Math.Max(blue, 0), 255);

                  lines[smoothFactor][w] = (byte)blue;
                  lines[smoothFactor][w + 1] = (byte)green;
                  lines[smoothFactor][w + 2] = (byte)red;

               }
            }
            bitmap.UnlockBits(bitmapData);
         }
      }

      public void Emboss(Bitmap bitmap, bool useSafe) {
         if (useSafe)
            EmbossSafe(bitmap);
         else
            EmbossUnsafe(bitmap);
      }
      
      private void EmbossSafe(Bitmap bitmap) {

         int embossFactor = 128;

         int dispW = 1;
         int dispH = 1;

         int w, h;

         for (w = 1; w < (bitmap.Width - 1); w++) {
            for (h = 1; h < (bitmap.Height - 1); h++) {
               Color pixel1 = bitmap.GetPixel(w, h);
               Color pixel2 = bitmap.GetPixel(w + dispW, h + dispH);

               int red = Math.Min(Math.Abs((pixel1.R) - (pixel2.R)) + embossFactor, 255);
               int green = Math.Min(Math.Abs((pixel1.G) - (pixel2.G)) + embossFactor, 255);
               int blue = Math.Min(Math.Abs((pixel1.B) - (pixel2.B)) + embossFactor, 255);

               red = Math.Min(Math.Max(red, 0), 255);
               green = Math.Min(Math.Max(green, 0), 255);
               blue = Math.Min(Math.Max(blue, 0), 255);

               bitmap.SetPixel(w, h, Color.FromArgb(red, green, blue));
            }
         }
      }

      private void EmbossUnsafe(Bitmap bitmap) {

         if (ValidatePixelFormat(bitmap) == false)
            return;

         int embossFactor = 128;
         int dispW = 1;
         int dispH = 1;
         int red, green, blue;

         unsafe {

            GetBitmapData(bitmap);

            BitmapData bitmapData = bitmap.LockBits(new Rectangle(0, 0, bitmap.Width, bitmap.Height), ImageLockMode.ReadWrite, bitmap.PixelFormat);

            byte* ptrFirstPixel = (byte*)bitmapData.Scan0;

            for (int h = Math.Abs(dispH); h < heightInPixels - Math.Abs(dispH); h++) {
               byte* currentLine = ptrFirstPixel + (h * bitmapData.Stride);
               byte* otherLine = ptrFirstPixel + ((h + dispH) * bitmapData.Stride);

               for (int w = (0 + dispW * bytesPerPixel); w < (widthInBytes - dispW * bytesPerPixel); w = w + bytesPerPixel) {
                  blue = Math.Min(Math.Abs((currentLine[w]) - (otherLine[w + dispW * bytesPerPixel])) + embossFactor, 255);
                  green = Math.Min(Math.Abs((currentLine[w + 1]) - (otherLine[w + 1 + dispW * bytesPerPixel])) + embossFactor, 255);
                  red = Math.Min(Math.Abs((currentLine[w + 2]) - (otherLine[w + 2 + dispW * bytesPerPixel])) + embossFactor, 255);

                  red = Math.Min(Math.Max(red, 0), 255);
                  green = Math.Min(Math.Max(green, 0), 255);
                  blue = Math.Min(Math.Max(blue, 0), 255);

                  currentLine[w] = (byte)(blue);
                  currentLine[w + 1] = (byte)(green);
                  currentLine[w + 2] = (byte)(red);
               }
            }
            bitmap.UnlockBits(bitmapData);
         }
      }

      public void Diffuse(Bitmap bitmap, bool useSafe) {
         if (useSafe)
            DiffuseSafe(bitmap);
         else
            DiffuseUnsafe(bitmap);
      }
      
      private void DiffuseSafe(Bitmap bitmap) {

         Random rnd = new Random();

         int diffuseW;
         int diffuseH;

         int w, h;

         for (w = 3; w < (bitmap.Width - 3); w++) {
            for (h = 3; h < (bitmap.Height - 3); h++) {
               diffuseW = rnd.Next(4) - 2;
               diffuseH = rnd.Next(4) - 2;

               int red = bitmap.GetPixel(w + diffuseW, h + diffuseH).R;
               int green = bitmap.GetPixel(w + diffuseW, h + diffuseH).G;
               int blue = bitmap.GetPixel(w + diffuseW, h + diffuseH).B;

               bitmap.SetPixel(w, h, Color.FromArgb(red, green, blue));
            }
         }
      }

      private void DiffuseUnsafe(Bitmap bitmap) {

         if (ValidatePixelFormat(bitmap) == false)
            return;

         Random rnd = new Random();

         int diffuseW;
         int diffuseH;

         int red, green, blue;

         unsafe {

            GetBitmapData(bitmap);

            BitmapData bitmapData = bitmap.LockBits(new Rectangle(0, 0, bitmap.Width, bitmap.Height), ImageLockMode.ReadWrite, bitmap.PixelFormat);

            byte* ptrFirstPixel = (byte*)bitmapData.Scan0;

            for (int h = 3; h < heightInPixels - 3; h++) {
               diffuseH = rnd.Next(4) - 2;

               byte* currentLine = ptrFirstPixel + (h * bitmapData.Stride);
               byte* otherLine = ptrFirstPixel + ((h + diffuseH) * bitmapData.Stride);

               for (int w = (0 + 3 * bytesPerPixel); w < (widthInBytes - 3 * bytesPerPixel); w = w + bytesPerPixel) {
                  diffuseW = rnd.Next(4) - 2;

                  blue = otherLine[w + diffuseW * bytesPerPixel];
                  green = otherLine[w + 1 + diffuseW * bytesPerPixel];
                  red = otherLine[w + 2 + diffuseW * bytesPerPixel];

                  currentLine[w] = (byte)(blue);
                  currentLine[w + 1] = (byte)(green);
                  currentLine[w + 2] = (byte)(red);
               }
            }
            bitmap.UnlockBits(bitmapData);
         }
       
      }

   }
}
